#include "Vista.h"

      SUBROUTINE ANNULUS(A,IMAGESR,IMAGEER,IMAGESC,IMAGEEC)

C---------------------------------------------------------------------------
C                                                                             
C XVista ANNULUS Command                            
C                                                                             
C Generates a radial brightness profile from an image by azimuthal averaging
C over a series of concentric annuli.  The annuli may be given an angle     
C of inclination and major axis position angle, where by the annuli are     
C then concentric ellipses, otherwise by default the annuli are round.      
C                                                                           
C The user must first either run AXES to compute the image centroid for the 
C annuli, or otherwise compute a centroid and enter it by hand via an       
C optional keyword                                                          
C                                                                           
C There is a provision to allow the user to use contours generated by the   
C PROFILE command for the azimuthal averaging via a keyword, otherwise,     
C ANNULUS provides a quick and dirty way of finding an average radial       
C profile of an object without taking into account actual isophotal contours
C at all.                                                                   
C
C In addition to computing the mean, it also computes the median around
C the annulus.  The MEDIAN keyword is provided to replace the azimuthal
C mean by the azimuthal median in the output profile buffer.
C
C By default, ANNULUS prints verbose output to the screen which may be
C redirected to an external file for further handling.  This output
C includes the full azimuthal statistics, not just the results put into
C the output buffer(s).  To suppress this output, the SILENT keyword
C is supplied.  
C
C If an annulus falls partially off the image, a warning message is
C printed (unless SILENT was given).  For applications where the 
C warning messages might interfere with downstream processing of the
C tty output from ANNULUS, the NOWARN keyword is provided to suppress
C warning messages.  
C
C --------------
C COMMAND SYNTAX                                                            
C                                                                           
C     ANNULUS dest source N=n [STEP=dr] [PA=pa] [INC=i] [CEN=r0,c0]         
C             [SCALE=s] [FAST] [PROF] [RAD=r] [SIG=buf] [MEDIAN]
C             [SILENT] [MASK] [LOADPROF]
C                                                                          
C where:                                                                    
C                                                                           
C     dest            Buffer to hold the average radial profile (spectrum)  
C                                                                           
C     source          Buffer with the image                                 
C                                                                           
C     N=n             Number of annuli to use starting at center            
C                                                                           
C     STEP=dr         (optional) Spacing between annuli.  If no scale is    
C                     given with the SCALE= keyword, the annulus spacing  
C                     is in units of pixels.                                
C                     The default value is 1.0 (single pixel steps).        
C                                                                           
C     PA=pa           (optional) Position Angle of Major Axis in degrees    
C                     measured Counterclockwise from the top of the image.  
C                     The default value is 0.0 (circular annuli).           
C                     If the optional PROF keyword is used, this keyword is 
C                     ignored.                                              
C                                                                           
C     INC=i           (optional) Inclination of Annuli in degrees defined as
C                     follows:                                              
C                             Face-On : i = 0 degrees                       
C                             Edge-On : i = 90 degrees                      
C                     i must be less than 90 degrees.                       
C                     The default value is 0.0 (circular annuli).           
C                     If the optional PROF keyword is used, this keyword is 
C                     ignored.                                              
C                                                                           
C     CEN=r0,c0       (optional) Center of annuli (in rows,colums on image).
C                     If not given, the program will use the object centroid
C                     computed by the most recent AXES command.  If AXES has
C                     not been run prior to running ANNULUS, the program    
C                     stop.                                                 
C                     If the optional PROF keyword is used, this keyword is 
C                     ignored.                                              
C                                                                           
C     SCALE=s         (optional) Image scale, typically expressed in units  
C                     of arcseconds/pixel.  The default value of s is 1.0.  
C                                                                           
C     FAST            (optional) Use faster bi-linear interpolation rather  
C                     than slower but more accurate Sinc interpolation.     
C                     Results will be screwy for small radii, so be warned. 
C                                                                           
C     PROF            (optional) Computes averages along contours defined by
C                     the contents of the PROFILE common block.  This uses  
C                     previously derived elliptical isophots and averages   
C                     along them.  The user must have previously used the   
C                     PROFILE command, or read in a SAVE`d profile with the 
C                     GET command.                                          
C                                                                           
C     RAD=r           (optional) Allows the user to compute an average along
C                     a single annulus.  In this case, the dest buffer    
C                     number is not required, or ignored if given.          
C
C     SIG=buf         (optional) also compute the standard deviation (sigma)
C                     around each annulus and put the vector with the sigma
C                     corresponding to each mean intensity level in buffer BUF
C
C     MEDIAN          (optional) substitute the median for the mean in the
C                     output buffer (annulus computes both, but only puts
C                     the mean in the output buffer by default)
C
C     SILENT          (optional) run silently with no screen output
C
C     NOWARN          (optional) suppress all warning messages, but allow
C                     all other screen output
C
C     MASK            (optional) use an image mask if available
C
C     LOADPROF        (optional) load the results into the PRF common block
C
C --------------------------------------------------------------------------
C                                                                           
C Author:   Rick Pogge     1987 March 13                                    
C                                                                           
C           Lick Observatory                                                
C           University of California                                        
C           Santa Cruz, CA  95064                                           
C                                                                           
C Compatability:   AXES and PROFILE                                         
C                  Output Redirection allowed                               
C                  VISTA version 3                                          
C                                                                           
C Modification History:                                                     
C   2002 July 1 - added SIG= keyword [rwp/osu]
C   2004 July 9 - Modified to correct the median calculation in the 
C                 case of incomplete annulli, and merged with the v7.01
C                 modifications (MASK and LOADPROF) [rwp/osu]
C    
C---------------------------------------------------------------------------

C The image to plot:

      REAL*4  A(IMAGESC:IMAGEEC,IMAGESR:IMAGEER)

C Communication with VISTA
#ifdef VMS
      INCLUDE 'VINCLUDE:VISTALINK.INC'
      INCLUDE 'VINCLUDE:IMAGELINK.INC'
      INCLUDE 'VINCLUDE:CUSTOMIZE.INC'
      INCLUDE 'VINCLUDE:PROFILE.INC'
      INCLUDE 'VINCLUDE:MASK.INC'
#else
      INCLUDE 'vistadisk/source/include/vistalink.inc'
      INCLUDE 'vistadisk/source/include/imagelink.inc'
      INCLUDE 'vistadisk/source/include/customize.inc'
      INCLUDE 'vistadisk/source/include/profile.inc'
      INCLUDE 'vistadisk/source/include/mask.inc'
#endif
C The AXES common block

      REAL*4  ECCAX
      REAL*4  ANGAX
      REAL*4  ANGMAJAX
      REAL*4  ANGMINAX
      REAL*4  XCAX, YCAX
      REAL*4  XPAX, YPAX
      
      COMMON /AX/ ECCAX, ANGAX, ANGMAJAX, ANGMINAX, XCAX, YCAX,
     &     XPAX, YPAX

C Previous PROFILE calculations are held in the PRF common block.  The
C parameters kept for each contour are as follows:
C
C      1       Contour central row number
C      2       Contour central column number
C      3       Position angle of contour, N at image top
C      4       Eccentricty of contour
C      5       Average surface brightness of contour
C      6       N=4 sine component of contour (average)
C      7       N=4 cosine component
C      8       N=2 sine component
C      9       N=2 cosine component
C     10       N=1 sine component
C     11       N=1 cosine component
C     12       Contour intensity derivative
C   NPRF       The number of contours
C  SCALE       The image scale arcsec/pixel

C  The maximum number of annuli

      INTEGER ANNMAX
      PARAMETER (ANNMAX = 500)

C Maximum number of samples along an annulus
C   this limit is set by utility/median.F, max number
C   of points that can be used for medians/%tile calculations

      INTEGER NSMAX
      PARAMETER (NSMAX = 10000)

C Arrays that use the WORK space (max 256*256=65536)

      REAL*4 AZAVG(ANNMAX)
      REAL*4 AZMED(ANNMAX)
      REAL*4 AZSIG(ANNMAX)
      REAL*4 AZPRF(NSMAX)
      COMMON /WORK/ AZAVG, AZSIG, AZMED, AZPRF

C Variables
      REAL*4  B
      REAL*4  C0
      REAL*4  COSTH
      REAL*4  COSINC
      REAL*4  CSAMP
      REAL*4  DCON
      REAL*8  DELT1
      REAL*4  DRANN
      REAL*4  DTHETA
      REAL*4  ECC
      REAL*4  F
      REAL*4  G(2)
      REAL*4  INC
      INTEGER ICS
      INTEGER IRS
      REAL*4  ISAMP
      REAL*4  ISCALE
      INTEGER NACTUAL
      INTEGER NANN
      INTEGER NCOL
      INTEGER NROW
      INTEGER NSAMP
      REAL*4  PA
      REAL*4  PHI
      REAL*4  PI
      REAL*4  PLOST
      REAL*4  R0
      REAL*4  RANN
      REAL*4  RPROJ
      REAL*4  RSAMP
      REAL*4  SININC
      REAL*4  SINTH
      REAL*4  THETA
      CHARACTER PARM*80
      REAL*4 SUMI
      REAL*4 SUMII
      REAL*4 AVG
      REAL*4 SIG
      REAL*4 ANNMED

C  Logical flags
      LOGICAL CENTER
      LOGICAL FAST
      LOGICAL KEYCHECK
      LOGICAL PROF
      LOGICAL SCALED
      LOGICAL SINGLE
      LOGICAL STEP
      LOGICAL SILENT
      LOGICAL NOWARN
      LOGICAL SAVESIG
      LOGICAL DOMEDIAN
      LOGICAL MASK
      LOGICAL MASKED
      LOGICAL LOADPROF

C Check keywords.  Exit if any are not understood.

      CALL KEYINIT
      CALL KEYDEF ('N=')
      CALL KEYDEF ('STEP=')
      CALL KEYDEF ('PA=')
      CALL KEYDEF ('INC=')
      CALL KEYDEF ('CEN=')
      CALL KEYDEF ('SCALE=')
      CALL KEYDEF ('RAD=')
      CALL KEYDEF ('FAST')
      CALL KEYDEF ('PROF')
      CALL KEYDEF ('PROF=')
      CALL KEYDEF ('SIG=')
      CALL KEYDEF ('MEDIAN')
      CALL KEYDEF ('SILENT')
      CALL KEYDEF ('NOWARN')
      CALL KEYDEF ('MASK')
      CALL KEYDEF ('LOADPROF')

C If the user has given a keyword that is not understood, exit, but also
C print a command syntax reminder

      IF (.NOT. KEYCHECK()) THEN
         XERR = .TRUE.
         PRINT *,'USAGE:'
         PRINT *,'ANNULUS dest source N=n [STEP=dr] [CEN=r0,c0] [PA=pa]'
         PRINT *,'    [INC=i] [SCALE=s] [FAST] [PROF] [RAD=r] [SIG=s]'
         PRINT *,'    [MEDIAN] [SILENT] [NOWARN] [MASK] [LOADPROF]'
         RETURN
      END IF

C Begin by defining center using output from AXES.  If AXES has not been
C run previously, then wait to see if the CEN= or PROF keywords have
C been invoked.

      IF ( XCAX .EQ. 0.0 .AND. YCAX .EQ. 0.0 ) THEN
         CENTER = .FALSE.
      ELSE
         CENTER = .TRUE.
         R0 = YCAX
         C0 = XCAX
      END IF

      PI = 3.1415926
      DCON = 1.745329e-2
      INC = 0.0
      ECC = 0.0
      PA  = 0.0
      NROW = IMAGEER - IMAGESR + 1
      NCOL = IMAGEEC - IMAGESC + 1
      STEP = .FALSE.
      DRANN = 1.0
      SCALED = .FALSE.
      ISCALE = 1.0
      PROF = .FALSE.
      IPRFCOL = 1
      SINGLE = .FALSE.
      FAST = .FALSE.
      SILENT = .FALSE.
      NOWARN = .FALSE.
      SAVESIG = .FALSE.
      DOMEDIAN = .FALSE.
      MASK = .FALSE.
      LOADPROF = .FALSE.

      ISBUF = 0

C  Read keywords

      DO 8701 I = 1, NCON
         
C   Number of annuli

         IF ( WORD(I)(1:2) .EQ. 'N=' ) THEN
            CALL ASSIGN ( WORD(I), F, PARM )
            IF (XERR) RETURN
            NANN = NINT(F)
            IF (NANN .LE. 0) THEN
               PRINT *,'The number of annuli must be >0'
               XERR    =.TRUE.
               RETURN
            END IF
            
            IF ( NANN .GT. ANNMAX ) NANN = ANNMAX
            
C   Radial space between successive annuli

         ELSE IF ( WORD(I)(1:5) .EQ. 'STEP=' ) THEN
            CALL ASSIGN ( WORD(I), F, PARM )
            IF (XERR) RETURN
            DRANN = F
            STEP = .TRUE.
            
C   Position angle of the major axis of elliptical annuli in degrees

         ELSE IF ( WORD(I)(1:3) .EQ. 'PA=' ) THEN
            CALL ASSIGN ( WORD(I), F, PARM )
            IF (XERR) RETURN
            PA = F
            
C   Inclination angle of annuli in degrees (0 = face-on)

         ELSE IF ( WORD(I)(1:4) .EQ. 'INC=' ) THEN
            CALL ASSIGN ( WORD(I), F, PARM )
            IF (XERR) RETURN
            INC = F
            IF ( INC .GE. 90.0 ) THEN
               PRINT *, 'Inclination must be < 90 degrees'
               XERR = .TRUE.
               RETURN
            ELSE IF ( INC. LT. 0.0 ) THEN
               PRINT *, 'Inclination must be >= 0 degrees'
               XERR = .TRUE.
               RETURN
            END IF
            COSINC = COSD(INC)
            SININC = SIND(INC)
            
C   User defined center for annuli

         ELSE IF ( WORD(I)(1:4) .EQ. 'CEN=' ) THEN
            CALL ASSIGNV ( WORD(I), 2, G, NG, PARM)
            IF (XERR) RETURN
            IF ( NG .NE. 2) THEN
               PRINT *, 'CEN= needs 2 arguments'
               XERR = .TRUE.
               RETURN
            END IF
            CENTER = .TRUE.
            R0 = G(1)
            C0 = G(2)
            IF ( R0 .GT. FLOAT(IMAGEER) .OR.
     &           R0 .LT. FLOAT(IMAGESR) ) THEN
               PRINT *, 'R0 outside image bounds'
               PRINT 
     &              '(1X,A,I5,A,I5,A)','(',IMAGESR,',',IMAGEER,')'
               XERR = .TRUE.
               RETURN
            END IF
            IF ( C0 .GT. FLOAT(IMAGEEC) .OR.
     &           C0 .LT. FLOAT(IMAGESC) ) THEN
               PRINT *, 'C0 outside image bounds'
               PRINT 
     &              '(1X,A,I5,A,I5,A)','(',IMAGESC, ',',IMAGEEC,')'
               XERR = .TRUE.
               RETURN
            END IF
            
C   Image Scale

         ELSE IF ( WORD(I)(1:6) .EQ. 'SCALE=' ) THEN
            CALL ASSIGN ( WORD(I), F, PARM )
            IF (XERR) RETURN
            ISCALE = F
            IF ( ISCALE .LE. 0.0 ) THEN
               PRINT *, 'SCALE must be >= 0.0'
               XERR = .TRUE.
               RETURN
            END IF
            SCALED = .TRUE.
            
C   Use faster bi-linear interpolation in outer annuli

         ELSE IF ( WORD(I) .EQ. 'FAST' ) THEN
            FAST = .TRUE.
            
C   Use the results of a previous PROFILE calculation to define
C   the annuli.

         ELSE IF ( WORD(I) .EQ. 'PROF' ) THEN
            PROF = .TRUE.
            CENTER = .TRUE.
            ISCALE = PRFSC(1)
            SCALED = .TRUE.
            IPRFCOL = 1
            
         ELSE IF ( WORD(I)(1:5) .EQ. 'PROF=' ) THEN
            CALL ASSIGN(WORD(I),F,PARM)
            IF (XERR) RETURN
            IPRFCOL = NINT(F)
            PROF = .TRUE.
            CENTER = .TRUE.
            ISCALE = PRFSC(IPRFCOL)
            SCALED = .TRUE.
            
C   Only average along a single, given annulus

         ELSE IF ( WORD(I)(1:4) .EQ. 'RAD=' ) THEN
            SINGLE = .TRUE.
            CALL ASSIGN ( WORD(I), F, PARM )
            IF (XERR) RETURN
            RANN = F
       
C   Also compute the sigma along the annulus

         ELSE IF (WORD(I)(1:4) .eq. 'SIG=') THEN
            CALL ASSIGN(WORD(I),F,PARM)
            IF (XERR) RETURN
            IF (F .NE. FLOAT(NINT(F))) THEN
               PRINT *,'ERROR: Invalid buffer for sigma vector'
               PRINT *,'       must be an integer'
               XERR = .TRUE.
               RETURN
            END IF
            ISBUF = NINT(F)
            SAVESIG = .TRUE.

C   Run without tty output (default is to print verbose output to tty)

         ELSE IF (WORD(I) .EQ. 'SILENT') THEN
            SILENT = .TRUE.

C   Suppress warning messages

         ELSE IF (WORD(I) .EQ. 'NOWARN') THEN
            NOWARN = .TRUE.

C   load the median into the output buffer instead of the mean
            
         ELSE IF (WORD(I) .EQ. 'MEDIAN') THEN
            DOMEDIAN = .TRUE.

C   Enable use of an image mask

         ELSE IF (WORD(I) .EQ. 'MASK') THEN
            MASK = .TRUE.

C   Load results into the PROFILE common
            
         ELSE IF (WORD(I) .EQ. 'LOADPROF') THEN
            LOADPROF = .TRUE.

         END IF
         
 8701 CONTINUE

C If the user has not defined an object center, either explicitly with the
C CEN= keyword, or by previous execution of either AXES or PROFILE, then
C inform the user of this and exit gracefully
      
      IF ( .NOT. CENTER ) THEN
         PRINT *, ' * * * No Object Center has been defined * * *'
         PRINT *, 'Either run AXES or use the CEN= keyword'
         PRINT *, '(or use PROFILE or a SAVEd profile calculation)'
         XERR = .TRUE.
         RETURN
      END IF

C  Check to see (if relevant) that the spectrum that is being created will be
C  in a different buffer from the image.

      IF ( .NOT. SINGLE ) THEN
         IF ( IBUF(1) .EQ. IBUF(2) ) THEN
            PRINT *,'The buffer for the destination spectrum must'
            PRINT *,'be different from the one holding the image.'
            XERR = .TRUE.
            RETURN
         END IF
      END IF

      IF (SAVESIG) THEN
         IF (ISBUF .EQ. IBUF(1) .OR. ISBUF .EQ. IBUF(2)) THEN
            PRINT *,'Cannot create sigma vector ',ISBUF
            PRINT *,'Image or new spectrum is already in that buffer.'
            XERR = .TRUE.
            RETURN
         END IF
      END IF

C Establish the spacing between annuli.  If the SCALE= and STEP= keywords
C were given, then convert the user defined annular spacing to pixels.  If
C the SCALE= keyword was not used, DRANN is already in units of pixels.

      IF ( SCALED .AND. STEP ) DRANN = DRANN / ISCALE

C Print calculation header.

      IF (.NOT. SILENT) THEN

         WRITE(olu, '(/1x,''* * * AZIMUTHAL AVERAGE CALCULATION * * *'')')

         IF (SINGLE) THEN
            WRITE(olu, 101) R0, C0, RANN*ISCALE
 101        FORMAT(/1X,'Single Annulus centered at (',f6.2,',',
     &           f7.2,'), with Radius =',f7.2 ) 
            IF (INC .NE. 0.0 ) 
     &           WRITE(olu, 102) PA, INC
 102        FORMAT(1X,'PA = ',f7.2,'degrees',
     &           ', Inclination = ',f5.2,' degrees')

         ELSE IF (PROF) THEN
            WRITE(olu, '(/1X,''Annuli defined from PROFILE fit'')' )

         ELSE
            WRITE(olu, 103) R0, C0
 103        FORMAT(/1X,'Annuli centered at (',f6.2,',',f7.2, ')') 
            IF (SCALED) THEN
               WRITE(olu, 104) DRANN*ISCALE, DRANN
 104           FORMAT(1X,'Annular spacing = ',f7.2, 
     &              ' Arcsec [',f7.2,' pixel(s)]') 
            ELSE
               WRITE(olu, 105) DRANN
 105           FORMAT(1X,'Annular spacing = ',f7.2,' Pixel(s)') 
            END IF
            
            IF (INC .NE. 0.0) 
     &           WRITE(olu, 106) PA, INC
 106        FORMAT(1X,'PA = ',f7.2,' degrees',
     &           ', Inclination = ',f5.2,' degrees') 
         END IF
         
      END IF
      
C   Initialize interpolation routines
      
      D = BINSET(A,NROW,NCOL,.FALSE.)

C********************************************************************************
C                                                                             *
C      In this section we do the azimuthal averaging along annuli             *
C                                                                             *
C********************************************************************************

C CASE I:  a single annulus (simple)

      IF (SINGLE) THEN
         IF ( RANN .LE. 2.0 ) THEN
            NSAMP = 8*PI*RANN + 4
         ELSE
            NSAMP = 2*PI*RANN + 1
         END IF
         NSAMP = MAX0(NSAMP,NSMAX)

         DTHETA = 2.0*PI / FLOAT(NSAMP)
         
         SUMI = 0.0
         SUMII = 0.0
         NACTUAL = 0
         
         DO 8702 I = 1, NSAMP
            IF (NOGO) RETURN
            THETA = FLOAT(I)*DTHETA
            COSTH = COS(THETA)
            SINTH = SIN(THETA)
            IF ( INC .EQ. 0.0 ) THEN
               RSAMP = R0 - ( RANN * COSTH )
               CSAMP = C0 - ( RANN * SINTH )
            ELSE
               PHI = DCON*PA + THETA
               RPROJ = RANN*COSINC/SQRT(1.0-(SININC*COSTH)**2)
               RSAMP = R0 - ( RPROJ * COS( PHI ) )
               CSAMP = C0 - ( RPROJ * SIN( PHI ) )
            END IF
            
            IRS = NINT(RSAMP)
            ICS = NINT(CSAMP)

            IF (MASK .AND. MASKED(ICS,IRS)) GOTO 8702

            IF (IRS .GE. IMAGESR .AND. IRS .LE. IMAGEER .AND.
     &          ICS .GE. IMAGESC .AND. ICS .LE. IMAGEEC ) THEN

               NACTUAL = NACTUAL + 1

               RSAMP = RSAMP - FLOAT(IMAGESR) + 1.0
               CSAMP = CSAMP - FLOAT(IMAGESC) + 1.0
               
               IF (RANN .LE. 15.0) THEN
                  ISAMP = XBIN(CSAMP,RSAMP,A,NROW,NCOL,.FALSE.)
               ELSE IF (FAST) THEN
                  ISAMP = BIN(CSAMP,RSAMP,A,NROW,NCOL,.FALSE.)
               ELSE
                  ISAMP = OUTBIN(CSAMP,RSAMP,A,NROW,NCOL,.FALSE.)
               ENDIF

               AZPRF(NACTUAL) = ISAMP
               SUMI = SUMI + ISAMP 
               SUMII = SUMII + (ISAMP*ISAMP)
            END IF
            
 8702    CONTINUE
         
         IF ( NACTUAL .LT. 1 ) THEN
            PRINT *, 'Given Annulus falls completely outside image'
            XERR = .TRUE.
            RETURN
            
         ELSE IF ( NACTUAL .LT. NSAMP ) THEN
            PLOST = 100.0*FLOAT(NACTUAL)/FLOAT(NSAMP)
            IF (.NOT. NOWARN) WRITE(olu,107) PLOST
 107        FORMAT(1X,'*** WARNING: Annulus is only',
     &           f5.1,'% complete.')
            
         END IF

C   Compute the average and sigma around the annulus.  We use the 
C   quick-and-dirty form of sigma here for computational convenience, 
C   even though it is only strictly valid in the case of large N

         CALL MEDIAN(AZPRF,NACTUAL,ANNMED,0.5)
         AVG = SUMI / FLOAT(NACTUAL)
         SIG = SQRT(SUMII/FLOAT(NACTUAL) - (AVG*AVG))

         WRITE(olu, '(/1X,''Azimuthal Average = '',1pg14.6)') AVG
         WRITE(olu, '(/1X,''          Median  = '',1pg14.6)') ANNMED
         WRITE(olu, '(/1X,''Azimuthal StDev   = '',1pg14.6)') SIG

         RETURN
         
C*****************************************************************************

C CASE II:  using contours from a previous PROFILE calculation

      ELSE IF (PROF) THEN

C     Output Format

 200     FORMAT (2X,F7.2,3X,F7.2,3X,F7.2,3X,F7.2,
     &           3X,F5.3,3(2X,1PG14.6))

C   Write Table Header Information

         IF (.NOT. SILENT) THEN
            WRITE(olu, 108)
 108        FORMAT(/1X,T4,'R (")',T16,'R0',T26,'C0',T36,'PA',
     &           T44,'Ecc',T54,'  Mean ',T64,'Median',T74,'Sigma')
            WRITE(olu, 109)
 109        FORMAT(1X,'--------------------','--------------------',
     &           '------------------------------------------------')
         END IF
         
C     First value is the "average" at the center

         NANN = NPRF(IPRFCOL) - 1
         RSAMP = PARAMPRF(1,1,IPRFCOL) - FLOAT(IMAGESR) + 1.0
         CSAMP = PARAMPRF(2,1,IPRFCOL) - FLOAT(IMAGESC) + 1.0
         
         AZAVG(1) = XBIN ( CSAMP, RSAMP, A, NROW, NCOL,.FALSE.)
         AZMED(1) = AZAVG(1)
         AZSIG(1) = 0.0

         IF (.NOT. SILENT) WRITE(olu, 200) 
     &           0.0,RSAMP,CSAMP,0.0,0.0,AZAVG(1),AZMED(1),0.0

C     Do each contour in turn

         DO 8703 J = 2, NPRF(IPRFCOL)
            IF (NOGO) RETURN
            
C     Grab relevant parameters of each contour

            R0 = PARAMPRF(1,J,IPRFCOL)
            C0 = PARAMPRF(2,J,IPRFCOL)
            PA = PARAMPRF(3,J,IPRFCOL)
            ECC = PARAMPRF(4,J,IPRFCOL)
            
            RANN = FLOAT(J)
            
            B = RANN * SQRT(1.0 - ECC*ECC)
            
            IF ( RANN .LE. 2.0 ) THEN
               NSAMP = 8*PI*RANN + 4
            ELSE
               NSAMP = 2*PI*RANN + 1
            END IF
            NSAMP = MAX0(NSAMP,NSMAX)
            
            DTHETA = 2.0*PI / FLOAT(NSAMP)
            
            SUMI = 0.0
            SUMII = 0.0
            NACTUAL = 0

            DO 8704 I = 1, NSAMP
               IF (NOGO) RETURN
               THETA = FLOAT(I)*DTHETA
               COSTH = COS(THETA)
               SINTH = SIN(THETA)
               IF ( ECC .EQ. 0.0 ) THEN
                  RSAMP = R0 - ( RANN * COSTH )
                  CSAMP = C0 - ( RANN * SINTH )
               ELSE
                  PHI = DCON*PA + THETA
                  RPROJ = B / SQRT(1.0 - (ECC*COSTH)**2)
                  RSAMP = R0 - ( RPROJ * COS( PHI ) )
                  CSAMP = C0 - ( RPROJ * SIN( PHI ) )
               END IF
               IRS = NINT(RSAMP)
               ICS = NINT(CSAMP)

               IF (MASK .AND. MASKED(ICS,IRS)) GOTO 8704

               IF ( IRS .GE. IMAGESR .AND. IRS .LE. IMAGEER .AND.
     &              ICS .GE. IMAGESC .AND. ICS .LE. IMAGEEC ) THEN

                  NACTUAL = NACTUAL + 1

                  RSAMP = RSAMP - FLOAT(IMAGESR) + 1.0
                  CSAMP = CSAMP - FLOAT(IMAGESC) + 1.0
                  
                  IF (RANN .LE. 15.0) THEN
                     ISAMP = XBIN(CSAMP,RSAMP,A,NROW,NCOL,.FALSE.)
                  ELSE IF ( FAST ) THEN
                     ISAMP = BIN(CSAMP,RSAMP,A,NROW,NCOL,.FALSE.)
                  ELSE
                     ISAMP = OUTBIN(CSAMP,RSAMP,A,NROW,NCOL,.FALSE.)
                  ENDIF
                  SUMI = SUMI + ISAMP
                  SUMII = SUMII + (ISAMP*ISAMP)
                  AZPRF(NACTUAL) = ISAMP
               END IF
 8704       CONTINUE
            
            IF ( NACTUAL .LT. 1 ) THEN
               AZAVG(J) = 0.0
               AZMED(J) = 0.0
               AZSIG(J) = 0.0
               IF (.NOT. SILENT) THEN
                  WRITE(olu, 200) RANN*ISCALE,R0,C0,PA,ECC,
     &                 AZAVG(J),AZMED(J),AZSIG(J)
                  IF (.NOT. NOWARN) WRITE(olu,110)
 110              FORMAT(4X,'* * *',1x,
     &                 'Annulus falls completely outside image * * * ')
               END IF
               
            ELSE IF ( NACTUAL .LT. NSAMP .AND. NACTUAL .GE. 1 ) THEN
               AVG = SUMI / FLOAT(NACTUAL)
               AZAVG(J) = AVG
               CALL MEDIAN(AZPRF,NACTUAL,ANNMED,0.5)
               AZMED(J) = ANNMED
               SIG = SQRT(SUMII/FLOAT(NACTUAL) - (AVG*AVG))
               AZSIG(J) = SIG
               PLOST = 100.0*FLOAT(NACTUAL)/FLOAT(NSAMP)
               IF (.NOT. SILENT) THEN
                  WRITE(olu, 200) RANN*ISCALE, R0, C0, PA, ECC,
     &                 AZAVG(J), AZMED(J), AZSIG(J)
                  IF (.NOT. NOWARN) WRITE(olu,111) PLOST
 111              FORMAT(1X,'*** WARNING: Annulus is only ',f5.1,
     &                 '% complete ***')
               END IF
               
            ELSE
               AVG = SUMI / FLOAT(NACTUAL)
               AZAVG(J) = AVG
               CALL MEDIAN(AZPRF,NACTUAL,ANNMED,0.5)
               AZMED(J) = ANNMED
               SIG = SQRT(SUMII/FLOAT(NACTUAL) - (AVG*AVG))
               AZSIG(J) = SIG
               
               IF (.NOT. SILENT) THEN
                  WRITE(olu, 200) RANN*ISCALE, R0, C0, PA, ECC,
     &                 AZAVG(J), AZMED(J), AZSIG(J)
               END IF

            END IF
            
C   Go get the next contour level until all gone

 8703    CONTINUE

C********************************************************************************

C CASE III:  Concentric Annuli defined by user inputs

      ELSE

C     Output Format

 300     FORMAT (2X,F7.2,3(3X,1PG14.6))

C     Write Header Information

         IF (.NOT. SILENT) THEN
            WRITE(olu, '(/1X,T4,''R (")'',T16,'' Mean  '',
     &           T26,''Median'',T36,''Sigma'')')
            WRITE(olu, '(1X,
     &      ''-----------------------------------------------'')' )
         END IF

C     First value is "average" at RANN=0.0 (at center)

         RSAMP = R0 - FLOAT(IMAGESR) + 1.0
         CSAMP = C0 - FLOAT(IMAGESC) + 1.0

         AZAVG(1) = XBIN ( CSAMP, RSAMP, A, NROW, NCOL, .FALSE. )
         AZMED(1) = AZAVG(1)
         AZSIG(1) = 0.0

         IF (.NOT. SILENT) 
     &        WRITE(olu, 300 ) 0.0, AZAVG(1), AZMED(1), 0.0

C     Do each annulus in turn

         DO 8705 J = 2, NANN
            
            IF (NOGO) RETURN
            
            RANN = FLOAT(J-1)*DRANN
            
            IF ( RANN .LE. 2.0 ) THEN
               NSAMP = 8*PI*RANN + 4
            ELSE
               NSAMP = 2*PI*RANN + 1
            END IF
            NSAMP = MAX0(NSAMP,NSMAX)
            
            DTHETA = 2.0*PI / FLOAT(NSAMP)
            
            SUMI = 0.0
            SUMII = 0.0
            NACTUAL = 0
            
            DO 8706 I = 1, NSAMP
               IF (NOGO) RETURN
               THETA = FLOAT(I)*DTHETA
               COSTH = COS(THETA)
               SINTH = SIN(THETA)
               IF ( INC .EQ. 0.0 ) THEN
                  RSAMP = R0 - ( RANN * COSTH )
                  CSAMP = C0 - ( RANN * SINTH )
                  
               ELSE
                  PHI = DCON*PA + THETA
                  RPROJ = RANN*COSINC/SQRT(1.0-(SININC*COSTH)**2)
                  RSAMP = R0 - ( RPROJ * COS( PHI ) )
                  CSAMP = C0 - ( RPROJ * SIN( PHI ) )
                  
               END IF
               
               IRS = NINT(RSAMP)
               ICS = NINT(CSAMP)

               IF (MASK .AND. MASKED(ICS,IRS)) GOTO 8706

               IF ( IRS .GE. IMAGESR .AND. IRS .LE. IMAGEER .AND.
     &              ICS .GE. IMAGESC .AND. ICS .LE. IMAGEEC ) THEN

                  NACTUAL = NACTUAL + 1
                  
                  RSAMP = RSAMP - FLOAT(IMAGESR) + 1.0
                  CSAMP = CSAMP - FLOAT(IMAGESC) + 1.0
                  
                  IF (RANN .LE. 15.0) THEN
                     ISAMP = XBIN(CSAMP,RSAMP,A,NROW,NCOL,.FALSE.)
                  ELSE IF ( FAST ) THEN
                     ISAMP = BIN(CSAMP,RSAMP,A,NROW,NCOL,.FALSE.)
                  ELSE
                     ISAMP = OUTBIN(CSAMP,RSAMP,A,NROW,NCOL,.FALSE.)
                  ENDIF
                  SUMI = SUMI + ISAMP
                  SUMII = SUMII + (ISAMP*ISAMP)

                  AZPRF(NACTUAL) = ISAMP

               END IF
               
 8706       CONTINUE
            
            IF ( NACTUAL .LT. 1 ) THEN
               AZAVG(J) = 0.0
               AZMED(J) = 0.0
               AZSIG(J) = 0.0
               IF (.NOT. SILENT) THEN
                  WRITE(olu, 300 ) RANN*ISCALE, AZAVG(J), 
     &                 AZMED(J), AZSIG(J)
                  IF (.NOT. NOWARN) WRITE(olu,112)
 112              FORMAT(4X,'* * *',1x,
     &                 'Annulus falls completely outside image * * * ')
               END IF
               
            ELSE IF ( NACTUAL .LT. NSAMP .AND. NACTUAL .GE. 1 ) THEN

               AVG = SUMI / FLOAT(NACTUAL)
               AZAVG(J) = AVG
               CALL MEDIAN(AZPRF,NACTUAL,ANNMED,0.5)
               AZMED(J) = ANNMED
               SIG = SQRT(SUMII/FLOAT(NACTUAL) - (AVG*AVG))
               AZSIG(J) = SIG
               PLOST = 100.0*FLOAT(NACTUAL)/FLOAT(NSAMP)
               IF (.NOT. SILENT) THEN
                  WRITE(olu, 300 ) RANN*ISCALE, AZAVG(J), AZMED(J), AZSIG(J)
                  IF (.NOT. NOWARN) WRITE(olu,113) PLOST
 113              FORMAT(1X,'*** WARNING: Annulus is only ',
     &                 f5.1,'% complete ***')
               END IF
               
            ELSE
               AVG = SUMI / FLOAT(NACTUAL)
               AZAVG(J) = AVG
               CALL MEDIAN(AZPRF,NACTUAL,ANNMED,0.5)
               AZMED(J) = ANNMED
               SIG = SQRT(SUMII/FLOAT(NACTUAL) - (AVG*AVG))
               AZSIG(J) = SIG
               IF (.NOT. SILENT) WRITE(olu, 300) 
     &              RANN*ISCALE,AZAVG(J),AZMED(J),AZSIG(J)
               
            END IF
            
 8705    CONTINUE
         
      END IF
      
C   Create the spectrum with the azimuthal averages in the specified
C   destination buffer.  If the SINGLE annulus option has been invoked, 
C   skip this step and exit.
      
      IF (SINGLE) RETURN
      
C   Load the appropriate FITS header cards
      
      IF (SCALED .AND. STEP) THEN
         DELT1 = DBLE(DRANN * ISCALE)
      ELSE IF ( SCALED .AND. .NOT. STEP ) THEN
         DELT1 = DBLE(ISCALE)
      ELSE
         DELT1 = DBLE(DRANN)
      END IF

C   Build a rudimentary FITS header

      TEMPHEAD = ' '
      CALL CCVSTRCPY(TEMPHEAD,HEADBUF(1,IM))
      CALL INHEADSET('NAXIS2',1,TEMPHEAD)
      CALL INHEADSET('NAXIS1',NANN,TEMPHEAD)
      CALL FHEADSET('CRVAL1',0.0D0,TEMPHEAD)
      CALL FHEADSET('CDELT1',DELT1,TEMPHEAD)
      CALL INHEADSET('CRPIX1',1,TEMPHEAD)
      IF ( SCALED ) THEN
         CALL CHEADSET('CTYPE1','Radius (Arcsec)',TEMPHEAD)
      ELSE
         CALL CHEADSET('CTYPE1','Radius (Pixels)',TEMPHEAD)
      END IF
      CALL CHEADSET('HISTORY','Output from ANNULUS',TEMPHEAD)

C   First, create the image for the azimuthal average (or median)

      CALL CHEADSET('BUNIT','Intensity',TEMPHEAD)

      IF (DOMEDIAN) THEN
         CALL CHEADSET('STATUS','Azimuthal Median',TEMPHEAD)
      ELSE
         CALL CHEADSET('STATUS','Azimuthal Average',TEMPHEAD)
      END IF
      CALL CREATEIM(LOCNEW,IMNEWSR,IMNEWER,IMNEWSC,IMNEWEC,1,
     &              TEMPHEAD,.TRUE.)
      IF (XERR) RETURN
      
      IF (DOMEDIAN) THEN
         CALL CCCOPIO (LOCNEW, 1, NANN, AZMED, 1, NANN, 0, 0 )
      ELSE
         CALL CCCOPIO (LOCNEW, 1, NANN, AZAVG, 1, NANN, 0, 0 )
      END IF

C     If LOADPROF was given on the command line, also load the PROFILE
C     common block
     
      IF (LOADPROF) THEN
         CALL CCVSTRCPY(PRFHEAD(1),HEADBUF(1,IM))
         CALL INHEADSET('NSTEPS',NANN,PRFHEAD(1))
         NPRF(1) = NANN
         DO I=1,NANN
            PARAMPRF(1,I,1) = R0
            PARAMPRF(2,I,1) = C0
            PARAMPRF(3,I,1) = PA
            PARAMPRF(4,I,1) = ECC
            IF (DOMEDIAN) THEN
               PARAMPRF(5,I,1) = AZMED(I)
               PARAMPRF(13,I,1) = AZMED(I)
            ELSE
               PARAMPRF(5,I,1) = AZAVG(I)
               PARAMPRF(13,I,1) = AZAVG(I)
            END IF
         END DO
      END IF

C   Second, if SIG= was used, create a second buffer with the sigma vector

      IF (SAVESIG) THEN
         IBUF(3) = ISBUF
         NINTS = 3
         CALL CHEADSET('BUNIT','Sigma',TEMPHEAD)
         CALL CHEADSET('STATUS','Azimuthal StDev',TEMPHEAD)
         CALL CREATEIM(LOC2,IMNEWSR,IMNEWER,IMNEWSC,IMNEWEC,3,
     &        TEMPHEAD,.TRUE.)
         IF (XERR) RETURN
         CALL CCCOPIO (LOC2, 1, NANN, AZSIG, 1, NANN, 0, 0 )
      END IF

      RETURN
      END
