#include "Vista.h"
	SUBROUTINE PHOTONS(A,ISROW,IEROW,ISCOL,IECOL)

C       Adds random noise and / or artificial stars to an image.

C       Keywords for the PHOTONS command:
C
C               MEAN=           gives the mean level (counts) added to
C                               the image. -1 if unspecified. If mean<0 then
C                               no photon noise is added to image. Otherwise
C                               you get photon noise. Recall, each time
C                               you run PHOTONS and add more stars, you''ll
C                               re-add photon noise unless you use mean<0
C               RN=             gives the readout noise (electrons)
C                               0 if unspecified. Again, only specify this
C                               once for a given image.
C               GAIN=           gives the conversion between photons
C                               and counts.  This is in the units
C                               photons per count. Needed to do noise
C                               properly.
C               NSTARS=         gives the number of stars to make.
C               FW=             star FWHM in pixels for Gaussian, or half
C                               width for arbitrary PSF (in disk file)
C               GAUSS           specifies gaussian image, else program
C                               will use 1-D PSF in file which must
C                               be specified with
C               FILE=           specifies file with N numbers N; (arr(i),i=1,N)
C                               representing 1-D PSF: half-width (no. of pixels
C                               per N bins) must be specified with FW keyword.
C                               If FILE is specified, GAUSS is ignored
C               PROF=           specifies file with 1-D psf, 2 columns, with
C                               radial distance and value in each line
C               NORM=           specifies normalization for polynomial
C                               images, 1 if not specified
C               COUNTS=a,b      gives ranges in total counts for the
C                               stars.
C               PHOT            store the positions and total counts of added
C                               stars in a photometry file.
C               NEW             creates a new photometry file.
C                               file.  NEW implies PHOT
C               AT=R,C          Put a single star at row R and column C
C                               This cannot be used with NSTARS.
C               DAOPSF=         Makes stars using a DAOPHOT PSF file
C               PSFLIB=         Makes stars using a PSFLIB file
C               PSFALL=         Use PSFLIB file with multiple PSFs for
C                               field dependence
C               NDIV            Allows user to change NDIV1 and NDIV2 the
C                               pixel subdivision for integration
C               TRUNC           Makes PHOTONS integer truncate
C               POISSON         Use true POISSON statistics
C               GALAXY          Use typical error distribution of galaxy images
C
C
C                               Donald Terndrup
C                               Lick Observatory
C                               University of California
C                               Santa Cruz, CA  96054
C                               October, 1984
C
C                               Version 3             May, 1985
C                               Version 4             September, 1985
C                                 added capacity to read PSF from disk file
C                                 changed handling of noise to use funct. ggnml
C                                                       J. Holtz.

C     Image
	REAL*4          A(ISCOL:IECOL,ISROW:IEROW)

C     Left side of keywords.
	CHARACTER       PARM*8
	CHARACTER*200    FILENAME, PSFFILE, STRFILE, FORMSTR
	CHARACTER*80    NAME, TWORD, CARD, LINE

C       Look-up table for generating star shapes
	REAL*4          STABLE(1000)
C       Range in counts
	REAL*4          COUNTS(2)
C       Location of star
	REAL*4          PIXEL(2), DX(9), DY(9)
	REAL*4          NORM, ALLVAL(5), RRR(5), ALLDVDX(5)
	REAL*4          SBMAG, IAVE, IERR
	REAL*8          DVAL, FHEAD

	LOGICAL         HAVECOUNTS, KNOWPOSN, HAVEGAUSS, HAVEFILE, HAVEDX
	LOGICAL         HAVEPHOT, NEW, PHOTNOISE, TRUNC, HAVEDIV, HAVEDY
	LOGICAL         KEYCHECK, SILENT, HAVEPSFLIB, POISS, DAOPHOT, HAVEPROF
	LOGICAL         HAVEKERNEL, HAVESTR, PSFALL, HAVESHR, PSFGRID, GALAXY  
        REAL            PSFVAL(0:1,0:1), PSFDVDX(0:1,0:1), PSFDVDY(0:1,0:1)
	INTEGER         UPPER, OPENC, READINT, GETNPAR, NLIB(2)
C	INTEGER*2       IDATA(4)
	REAL            ADATA(4), PROF(1000,2)

C       Include files.
#ifdef VMS
	INCLUDE 'VINCLUDE:IMAGELINK.INC'               ! Image parameters
	INCLUDE 'VINCLUDE:VISTALINK.INC'               ! Communication with VISTA
	INCLUDE 'VINCLUDE:PHOTFILES.INC'               ! Photometry files
	INCLUDE 'VINCLUDE:TIMEDATA.INC'                ! Properties of the image
	INCLUDE 'VINCLUDE:PSFLIB.INC'
	INCLUDE 'VINCLUDE:CUSTOMIZE.INC'
#else
	INCLUDE 'vistadisk/source/include/imagelink.inc'
	INCLUDE 'vistadisk/source/include/vistalink.inc'
	INCLUDE 'vistadisk/source/include/photfiles.inc'
	INCLUDE 'vistadisk/source/include/timedata.inc'
	INCLUDE 'vistadisk/source/include/psflib.inc'
	INCLUDE 'vistadisk/source/include/customize.inc'
#ifdef __DECSTA
	INTEGER DECTIME
#else
	INTEGER TIME
#endif
#endif
C       The components of a photometry file:

	CHARACTER*80    PHOTLABEL
	REAL*4          PHOTDATA(NUMDATA)

C       Constants.

	PARAMETER (PI = 3.141592654)
	PARAMETER (TWOPI = 2.0 * PI)
	PARAMETER (CONV = 180.0 / PI * 3600.0 / 15.0)

C       Include stuff for DAOPSF software

	LOGICAL HAVEDAOPSF
	PARAMETER (MAXPSF = 301) 
	REAL*4 GAUSS(10), PSF(MAXPSF,MAXPSF)
	COMMON /WORK/ PSF, STABLE

	DATA            PHOTLABEL /' '/
	DATA            PHOTDATA /NUMDATA * 0.0/

C       List allowed keywords.

	CALL KEYINIT
	CALL KEYDEF('MEAN=')
	CALL KEYDEF('GAIN=')
	CALL KEYDEF('NSTARS=')
	CALL KEYDEF('FW=')
	CALL KEYDEF('COUNTS=')
	CALL KEYDEF('PHOT')
	CALL KEYDEF('NEW')
	CALL KEYDEF('AT=')
	CALL KEYDEF('GAUSS')
	CALL KEYDEF('NORM=')
	CALL KEYDEF('RN=')
	CALL KEYDEF('FILE=')
	CALL KEYDEF('PROF=')
	CALL KEYDEF('KERNEL=')
	CALL KEYDEF('STR=')
	CALL KEYDEF('SHR=')
	CALL KEYDEF('NCOLOR=')
	CALL KEYDEF('NPAR=')
	CALL KEYDEF('DAOPSF=')
	CALL KEYDEF('PSFLIB=')
	CALL KEYDEF('PSFALL=')
	CALL KEYDEF('PSFGRID=')
	CALL KEYDEF('TRUNC')
	CALL KEYDEF('NDIV')
	CALL KEYDEF('SILENT')
	CALL KEYDEF('POISSON')
	CALL KEYDEF('GALAXY')
	CALL KEYDEF('DX=')
	CALL KEYDEF('DY=')
	CALL KEYDEF('BUFFER=')

C       Compute the square-root of 2 pi.

	SQ2PI = SQRT(TWOPI)

C       Other initializations.

	AVERAGE  = -1.
	RN = 0.0
	CONVERSN = 0.0
	NTOPUT = 0
	NORM = 1.
	NPAR = 0
	STARFW = 5.
	NDIV1 = 7
	NDIV2 = 3
	NCOLOR = 1

C       Check for the parameters.

	HAVECOUNTS   = .FALSE.
	HAVEPHOT     = .FALSE.
	KNOWPOSN     = .FALSE.
	HAVEGAUSS    = .FALSE.
	HAVEFILE     = .FALSE.
	HAVEPROF     = .FALSE.
	NEW           = .FALSE.
	PHOTNOISE    = .TRUE.
	HAVEKERNEL   = .FALSE.
	HAVESTR      = .FALSE.
	HAVESHR      = .FALSE.
	HAVEDAOPSF  = .FALSE.
	HAVEPSFLIB  = .FALSE.
	PSFALL        = .FALSE.
	PSFGRID        = .FALSE.
	TRUNC         = .FALSE.
	HAVEDIV      = .FALSE.
	SILENT       = .FALSE.
	POISS        = .FALSE.
        GALAXY       = .FALSE.
        HAVEDX = .FALSE.
        HAVEDY = .FALSE.
        BUFFER = 0

	DO 8701 I=1,NCON
		TWORD = WORD(I)
		L = UPPER(TWORD)
		IF (TWORD .EQ. ' ') THEN
			GOTO 999

C               Photometry file.

		ELSE IF (TWORD(1:4) .EQ. 'PHOT' ) THEN
			HAVEPHOT = .TRUE.

C               New photometry file.

		ELSE IF (TWORD .EQ. 'NEW' ) THEN
			HAVEPHOT = .TRUE.
			NEW       = .TRUE.

C               Average level of background.

		ELSE IF (TWORD(1:5) .EQ. 'MEAN=' ) THEN
			CALL ASSIGN(TWORD,AVERAGE,PARM)
			IF (XERR) GOTO 9999

C               Readout noise.

		ELSE IF (TWORD(1:3) .EQ. 'RN=' ) THEN
			CALL ASSIGN(TWORD,RN,PARM)
			IF (XERR) GOTO 9999

C               Photon-per-count parameter.

		ELSE IF (TWORD(1:5) .EQ. 'GAIN=' ) THEN
			CALL ASSIGN(TWORD,CONVERSN,PARM)
			IF (XERR) GOTO 9999

C               Number of stars.

		ELSE IF (TWORD(1:7) .EQ. 'NSTARS=' ) THEN
			CALL ASSIGN(TWORD,TEMP,PARM)
			IF (XERR) GOTO 9999
			NTOPUT = INT(TEMP)

C               Location of a single star.

		ELSE IF (TWORD(1:3) .EQ. 'AT=' ) THEN
			CALL ASSIGNV(TWORD,2,PIXEL,NF,PARM)
			IF (XERR) RETURN

			IF (NF .NE. 2) THEN
				PRINT *,'Syntax is AT=r,c'
				XERR = .TRUE.
				RETURN
			END IF

			KNOWPOSN = .TRUE.
			NTOPUT = 1

C               Star full-width.

		ELSE IF (TWORD(1:3) .EQ. 'FW=' ) THEN
			CALL ASSIGN(TWORD,STARFW,PARM)
			IF (XERR) GOTO 9999

		ELSE IF (TWORD(1:5) .EQ. 'NPAR=' ) THEN
			CALL ASSIGN(TWORD,TEMP,PARM)
			NPAR = INT(TEMP)
			IF (XERR) GOTO 9999

C               Flag for gaussian image

		ELSE IF (TWORD(1:3) .EQ. 'GAU' ) THEN
			HAVEGAUSS = .TRUE.

		ELSE IF (TWORD(1:5) .EQ. 'TRUNC' ) THEN
			TRUNC = .TRUE.

		ELSE IF (TWORD(1:6) .EQ. 'SILENT') THEN
			SILENT = .TRUE.

		ELSE IF (TWORD(1:7) .EQ. 'POISSON') THEN
			POISS = .TRUE.

	        ELSE IF (TWORD(1:6) .EQ. 'GALAXY') THEN
		        GALAXY = .TRUE.

		ELSE IF (TWORD(1:4) .EQ. 'NDIV' ) THEN
			PRINT '(1X,''ENTER NDIV1,NDIV2: ''$)'
			READ *, NDIV1,NDIV2

		ELSE IF (TWORD(1:4) .EQ. 'STR=' ) THEN
			HAVESTR = .TRUE.
			STRFILE = WORD(I)(5:)
			CALL FILEDEF(STRFILE,NAME,DATADIR,'.str')
			OPEN(3,FILE=NAME,STATUS='OLD',IOSTAT=IOS)
			IF (IOS .NE. 0) THEN
			  PRINT *, ' ERROR OPENING STR FILE '
			  PRINT *, ' IOSTAT = ', IOS
			  RETURN
			END IF
			REWIND(3)
			NTOPUT = 1

		ELSE IF (TWORD(1:4) .EQ. 'SHR=' ) THEN
			HAVESTR = .TRUE.
			HAVESHR = .TRUE.
			STRFILE = WORD(I)(5:)
			CALL FILEDEF(STRFILE,NAME,DATADIR,'.shr')
			L = INDEX (NAME,' ')
			NAME(L:L) = CHAR(0)
			ISTAT = OPENC(JFILE,NAME,0)
			IF (ISTAT .LT. 0) THEN
			  PRINT *, ' ERROR OPENING SHR FILE '
			  PRINT *, ' ISTAT = ', ISTAT
			  RETURN
			END IF
			ISTAT = READINT(JFILE,ADATA,16)
#ifndef __DECSTA
			ISTAT = PACKFIT4(ADATA,ADATA,16)
#endif
			IF (ISTAT .LT. 0) THEN
			  PRINT *, ' ERROR READING SHR FILE '
			  XERR = .TRUE.
			  RETURN
			END IF
			ACOL = ADATA(1)
			AROW = ADATA(2)
			A1 = ADATA(3)
			A2 = ADATA(4)
			NTOPUT = 1

C       NCOLOR= keyword tells which column in STR file to use for counts
		ELSE IF (TWORD(1:7) .EQ. 'NCOLOR=' ) THEN
			CALL ASSIGN(TWORD,TEMP,PARM)
			IF (XERR) GOTO 9999
			NCOLOR = INT(TEMP)

C               File for 1-D PSF

		ELSE IF (TWORD(1:5) .EQ. 'FILE=' ) THEN
			FILENAME = WORD(I)(6:)
			HAVEGAUSS = .FALSE.
			HAVEFILE  = .TRUE.

		ELSE IF (TWORD(1:5) .EQ. 'PROF=' ) THEN
			FILENAME = WORD(I)(6:)
			HAVEGAUSS = .FALSE.
			HAVEPROF  = .TRUE.

		ELSE IF (TWORD(1:7) .EQ. 'DAOPSF=' ) THEN
			PSFFILE = WORD(I)(8:)
			HAVEDAOPSF  = .TRUE.
			CALL FILEDEF(PSFFILE,NAME,DAODIR,'.psf')
			OPEN (UNIT=21,FILE=NAME,STATUS='OLD',
     &                          IOSTAT=IOS)
			IF (IOS .NE. 0) THEN
			  PRINT *, ' ERROR OPENING DAOPSF FILE '
			  PRINT *, ' IOSTAT = ', IOS
			  RETURN
			END IF
			REWIND(21)
			PRINT *, ' Using IPSFMODE: ', IPSFMODE
			NPAR = GETNPAR(IPSFMODE)
			WRITE(FORMSTR,210) NPAR-1
210                     FORMAT ('(2(I3,1X), 2(1PE14.6),',I12.12,
     &                          '(1PE14.6), 2(1X,0PF7.2))')
C210                     FORMAT('(2I3,2F10.3,',I2.2,'(F10.6),2F7.2)')
			READ(21,FORMSTR) NPSF,NPSF,PSFMAG,(GAUSS(II),
     &                        II=1,NPAR),XPSF,YPSF
C  210                  FORMAT(2I3,2F10.3,<NPAR-1>F10.6,2F7.2)
			IF (NPSF .GT. MAXPSF) THEN
			  PRINT *, 'PSF is too big for current ',
     &                             'array size'
			  RETURN
			END IF
			READ(21,211) ((PSF(II,JJ),II=1,NPSF),JJ=1,NPSF)
211                     FORMAT (10E13.6)
			CLOSE(21)
			STARFW = NPSF / 4.

		ELSE IF (TWORD(1:7) .EQ. 'PSFLIB=' .OR.
     &                   TWORD(1:7) .EQ. 'PSFALL=' .OR.
     &                   TWORD(1:8) .EQ. 'PSFGRID=') THEN
                        L = INDEX(WORD(i),'=')
			PSFFILE = WORD(I)(L+1:)
			HAVEPSFLIB  = .TRUE.
			CALL FILEDEF(PSFFILE,NAME,PSFDIR,'.lib')
                        ifile = -1
			call rdfitshead(temphead,name,ifile,ierr,.true.)
			if (ierr .ne. 0) then
			  print *, 'Error opening PSFLIB file', NAME
			  return
			end if
			npsflib = inhead('NAXIS1',temphead)
			nlib(1) = inhead('NLIB',temphead)
			nside = npsflib / nlib(1)
			psftot = sngl(fhead('PSFTOT',temphead))
			nall = 1
                        nsamp = 1
			if (tword(1:7) .eq. 'PSFALL=') then
			  nall = inhead('NAXIS3',temphead)
			  psfall = .true.
			end if
			if (tword(1:8) .eq. 'PSFGRID=') then
			  nall = inhead('NAXIS3',temphead)
			  psfgrid = .true.
                          x0psf = inhead('X0',temphead)
                          y0psf = inhead('Y0',temphead)
                          dxpsf = inhead('DX',temphead)
                          dypsf = inhead('DY',temphead)
                          nsamp = inhead('NAXIS4',temphead)
                          if (nsamp .le. 0) nsamp = 1
                          if (nsamp .gt. 1) then
                            do j=1,nsamp
                              write(card,145) j
145                           format('NLIB',i1)
                              nlib(j) = inhead(card,temphead)
                            end do
                          end if
                        end if

			nbytes = 4*npsflib*npsflib
			  
			if (nlib(1) .le. 0 .or. psftot .le. 0) goto 575
			call ccalloc(nbytes*nall*nsamp,location)
			call ccrdfits(location,npsflib*nall*nsamp,npsflib,ifile,
     &                      temphead,ierr)
			if (ierr .ne. 0) goto 575
#ifdef VMS
			close(ifile)
#else
			istat = closec(ifile)
#endif

C               PSF Normalization

		ELSE IF (TWORD(1:5) .EQ. 'NORM=' ) THEN
			CALL ASSIGN(TWORD,NORM,PARM)
			IF (XERR) GOTO 9999

		ELSE IF (TWORD(1:7) .EQ. 'KERNEL=' ) THEN
			CALL ASSIGN(TWORD,RKER,PARM)
			IF (XERR) GOTO 9999
			HAVEKERNEL = .TRUE.
			KERNEL = INT(RKER)

C               Distrubution of total brightnesses.

		ELSE IF (TWORD(1:7) .EQ. 'COUNTS=' ) THEN
			CALL ASSIGNV(TWORD,2,COUNTS,NFOUND,PARM)
			IF (XERR) GOTO 9999

			IF (NFOUND .EQ. 1) THEN
				COUNTS(2) = COUNTS(1)
				COUNTS(1) = 0
			END IF

			HAVECOUNTS = .TRUE.

		ELSE IF (TWORD(1:3) .EQ. 'DX=' ) THEN
			CALL ASSIGNV(TWORD,8,DX,NF,PARM)
			IF (XERR) RETURN
                        IF (NF .NE. 8) THEN
                           PRINT *, 'Incorrect no of dx parameters'
                           XERR = .TRUE.
                           RETURN
                        END IF
                        HAVEDX = .TRUE.

		ELSE IF (TWORD(1:3) .EQ. 'DY=' ) THEN
			CALL ASSIGNV(TWORD,8,DY,NF,PARM)
			IF (XERR) RETURN
                        IF (NF .NE. 8) THEN
                           PRINT *, 'Incorrect no of dx parameters'
                           XERR = .TRUE.
                           RETURN
                        END IF
                        HAVEDY = .TRUE.

		ELSE IF (TWORD(1:7) .EQ. 'BUFFER=' ) THEN
			CALL ASSIGN(TWORD,BUFFER,PARM)
                        IF (XERR) RETURN

		END IF
8701    CONTINUE
 999    CONTINUE

C       Exit if the user gave a keyword we do not understand.

	IF (.NOT. KEYCHECK()) THEN
		XERR = .TRUE.
		RETURN
	END IF

C       Open the PSF file

	IF ( .NOT. HAVEGAUSS .AND. .NOT. HAVEDAOPSF .AND.
     &    .NOT. HAVEPSFLIB .AND. NTOPUT .GT. 0) THEN

	   IF (HAVEFILE .OR. HAVEPROF) THEN
		OPEN(UNIT=22,FILE=FILENAME,STATUS='OLD',IOSTAT=IOS)
	   ELSE
		CALL FILEDEF('psf',FILENAME,PSFDIR,'.pho')
		OPEN(UNIT=22,FILE=FILENAME,STATUS='OLD',IOSTAT=IOS)
		PRINT * ,' USING PSF FROM V$PSFDIR:PSF.PHO,FW=',
     &                      STARFW
	   END IF

	   IF (IOS .NE. 0) THEN
		PRINT *, ' ERROR OPENING 1-D PSF FILE '
		PRINT *, ' IOSTAT = ', IOS
		RETURN
	   END IF
	END IF

C       If the mean is less than zero, set for no photon noise.

	IF (AVERAGE .LT. 0.0) THEN
		PHOTNOISE = .FALSE.
		AVERAGE = 0.
	END IF

C       If the number of stars is less than zero, exit.

	IF (NTOPUT .LT. 0 ) THEN
		PRINT *,'Number of stars must be positive.'
		XERR = .TRUE.
		GOTO 9999
	END IF

C       If the user gave both AT= and NSTARS=, exit.

	IF (KNOWPOSN .AND. (NTOPUT .GT. 1 .OR. HAVESTR)) THEN
		PRINT *,'Cannot use AT= and NSTARS= or STR= together.'
		XERR = .TRUE.
		RETURN
	END IF

C       If we have a position, check it.

	IF (KNOWPOSN) THEN
		IF (PIXEL(1) .LT. FLOAT(ISROW) .OR.
     &              PIXEL(1) .GT. FLOAT(IEROW) .OR.
     &              PIXEL(2) .LT. FLOAT(ISCOL) .OR.
     &              PIXEL(2) .GT. FLOAT(IECOL)) THEN
			PRINT *,'Location of star outside bounds ',
     &                          'of image.'
			XERR = .TRUE.
			RETURN
		END IF

		NTOPUT = 1
	END IF

C       Initialize photometry file if needed.
C       Get properties of the image.

	IF (HAVEPHOT) THEN
		IF (NEW) THEN
			IF (NSTARS .NE. 0) CALL CLOSEPHOT
			CALL OPENPHOT
			NSTARS = 0
		END IF
		AM = 0.
		HJD = 2444000.
                CALL CCVSTRCPY(TEMPHEAD,HEADBUF(1,IM))
		CALL GETHJD(TEMPHEAD,XERR)
	END IF

C       If there are stars, make sure we have counts.  The counts
C       must be positive.

	IF (NTOPUT .NE. 0 .AND. .NOT. HAVESTR) THEN
		IF (.NOT. HAVECOUNTS) THEN
			PRINT *,'If you want stars added to the ',
     &                          'image, you must'
			PRINT *,'give their distribution of ',
     &                          'brightnesses.'
			PRINT *,'Use the COUNTS keyword.'
			XERR = .TRUE.
			GOTO 9999
		ELSE
			IF (HAVECOUNTS) THEN
				IF (COUNTS(1) .LT. 0.0 .OR.
     &                              COUNTS(2) .LT. 0.0) THEN
				    PRINT *,'The counts given must ',
     &                                      'be positive.'
				    XERR = .TRUE.
				    GOTO 9999
				END IF
			END IF
		END IF

C               If the full width for stars is <= 0, set it to 2.0.

		IF (STARFW .LE. 0.0) STARFW = 2.0
	END IF

C       Find a kernel for the random number generator.

	IF (.NOT. HAVEKERNEL) THEN
#ifdef VMS
		KERNEL = FOR$SECNDS(0)
#else
#ifdef __DECSTA
		KERNEL = DECTIME()
#else
		KERNEL = TIME()
#endif
#endif
		TEST = KERNEL / 2. - INT(KERNEL/2.)
		IF ( TEST .LT. 0.1) KERNEL = KERNEL + 1
	END IF
	IF (KERNEL .GT. 0) KERNEL=-1*KERNEL
C	WRITE(olu,*) ' KERNEL USED: ', KERNEL

C       If the conversion factor is <= 0, set it to 1.

	IF (CONVERSN .LE. 0.0) CONVERSN = 1.0

C       We come here to add stars to the image.

C       If there aren''t any stars, go to add sky or readout noise

	IF (HAVESTR) GOTO 287

	IF (NTOPUT .EQ. 0) GOTO 2999

C       Compute the square of the full width.

287     STARFWSQ = STARFW ** 2

C       Fill STABLE with the profile for the star.  STABLE(I) is the
C       value of the normalized star profile at R = (I - 1) / 50. * STARFW.
C       Thus the profile extends to 5 full widths of the star.
C       We translate from pixels to bins by the formula:
C
C                       PIXELS = BINS / BINCONV
C                       (I Say:  BINS * BINCONV   Holtz.)

	IF (HAVEGAUSS) THEN

		BINCONV   = STARFW / 50.0

		SIGMA = STARFW / 2.35482
		CON = 0.5 / PI / SIGMA**2
		DO 8702 I=1,250
			R = FLOAT(I - 1) * BINCONV
			STABLE(I) = CON * EXP(-0.5 * (R / SIGMA)**2)
8702            CONTINUE
		NTABLE = 250

	ELSE IF (.NOT. HAVEDAOPSF .AND. .NOT. HAVEPSFLIB) THEN

		IF (NOGO) RETURN

                IF (HAVEFILE) THEN
		  READ(22,*) NTABLE
		  IF (NTABLE .GT. 1000) THEN
		    PRINT *, 'PSF table too big'
		    RETURN
		  END IF
		  READ(22,*) (STABLE(I),I=1,NTABLE)
		  STABLE (NTABLE) = 0.
		  CLOSE (22)
		  BINCONV  =  STARFW / (NTABLE-1)
                ELSE IF (HAVEPROF) THEN
	          N = 1
101	          READ(22,*,END=199) PROF(N,1), PROF(N,2)
                  N = N + 1
                  GOTO 101
199               N = N - 1
	          DO I=1,1000
		    R = FLOAT(I - 1) / N * STARFW
                    CALL ALININT(PROF(1,1),PROF(1,2),N,R,STABLE(I))
                  END DO
                END IF

	END IF

C       Note the size of the image.

	NROW = IEROW - ISROW + 1
	NCOL = IECOL - ISCOL + 1

C       Write a label.

	WRITE(olu,*) '    Row         Column       NCNTS     SUMHERE'

C       For the stars we are to place on the image...
	RANGE = COUNTS(2) - COUNTS(1)

C
C       If we have a star file, set up position and counts

        DAOPHOT = .FALSE.

2001    IF (HAVESTR) THEN
	   NTOPUT = 1
	   IF (HAVESHR) THEN
	     ISTAT = READINT ( JFILE, ADATA, 16)
	     IF (ISTAT .LT. 0) GOTO 2999
#ifdef MSBFirst
             ISTAT = PACKFIT4(ADATA,ADATA,16)
#endif
	     PIXEL(1) = ADATA(1)
	     PIXEL(2) = ADATA(2)
	     RRR(1) = ADATA(3)
	     RRR(2) = ADATA(4)
C	     PIXEL(1) = (IDATA(2) + 32768.)/65536. * AROW
C	     PIXEL(2) = (IDATA(1) + 32768.)/65536. * ACOL
C	     RRR(1) = (IDATA(3) + 32768.)/65536. * A1
C	     RRR(2) = (IDATA(4) + 32768.)/65536. * A2
	   ELSE
	     READ(3,'(A)',END=2999,ERR=2001) LINE
             IF (INDEX(LINE,'NL') .GT. 0) THEN
               DAOPHOT = .TRUE.
               READ(3,*)
               READ(3,*)
	       READ(3,'(A)',END=2999,ERR=2001) LINE
             END IF
             READ(LINE,*,ERR=2001) III,PIXEL(2),PIXEL(1),(RRR(JJ),JJ=1,NCOLOR)
             IF (DAOPHOT) RRR(NCOLOR) = 10.**(-0.4*(RRR(NCOLOR)-25))
	   END IF
	   RNCNTS = RRR(NCOLOR)
	   IF (PIXEL(1) .LT. FLOAT(ISROW) .OR.
     &         PIXEL(1) .GT. FLOAT(IEROW) .OR.
     &         PIXEL(2) .LT. FLOAT(ISCOL) .OR.
     &         PIXEL(2) .GT. FLOAT(IECOL)) THEN
	        IF (.NOT. SILENT) 
     &            PRINT *,'Location of star outside bounds of image.'
		GOTO 2001
	   END IF
	   KNOWPOSN = .TRUE.
	   IF (STARFW .EQ. 0) THEN
		IF (RNCNTS .GT. 10000.) THEN
		   STARFW = 7
		ELSE IF (RNCNTS .GT. 1000.) THEN
		   STARFW = 5
		ELSE IF (RNCNTS .GT. 15.) THEN
		   STARFW = 4
		ELSE IF (RNCNTS .GT. 5.) THEN
		   STARFW = 3
		ELSE
		   STARFW = 2
		END IF
	    END IF

	END IF
	GOTO 2003

2003    CONTINUE

C       We will integrate the probability distribution for the star
C               over a box which extends for 5 full widths on either side.
C               Set the limits for the box.
        IF (HAVEGAUSS) THEN
	    DIFF = 5.0 * STARFW
	ELSE
	    DIFF = STARFW
	END IF

	DO 8703 I=1,NTOPUT

C               Return if the user typed control-C

		IF (NOGO) RETURN
#ifdef __X11
                CALL LOOPDISP
#endif

		SUMHERE = 0.0

C               Pick a random row and column location or use the one we
C               have been given.

		IF (KNOWPOSN) THEN
		  ROW = PIXEL(1)
		  COL = PIXEL(2)
		ELSE
		  ROW = RAN1(KERNEL) * (FLOAT(NROW)-2*BUFFER) + 
     &                  FLOAT(ISROW) + BUFFER
		  COL = RAN1(KERNEL) * (FLOAT(NCOL)-2*BUFFER) + 
     &                  FLOAT(ISCOL) + BUFFER
		END IF
                IF (HAVEDX) 
     &             COL = COL + DX(1)+DX(2)*(ROW-DX(7))+DX(3)*(COL-DX(8))
     &                      + DX(4)*(ROW-DX(7))**2 + DX(5)*(COL-DX(8))**2
     &                      + DX(6)*(ROW-DX(7))*(COL-DX(8))
                IF (HAVEDY) 
     &             ROW = ROW + DY(1)+DY(2)*(ROW-DY(7))+DY(3)*(COL-DY(8))
     &                      + DY(4)*(ROW-DY(7))**2 + DY(5)*(COL-DY(8))**2
     &                      + DY(6)*(ROW-DY(7))*(COL-DY(8))

C               Pick a brightness for this star.

		IF (.NOT. HAVESTR) THEN
		  ARAN = RAN1(KERNEL)
		  RNCNTS = ARAN * RANGE + COUNTS(1)
		END IF

		JRLOW = NINT(ROW - DIFF)
		JRHI  = NINT(ROW + DIFF)
		ICLOW = NINT(COL - DIFF)
		ICHI  = NINT(COL + DIFF)

		IF (JRLOW .LT. ISROW) JRLOW = ISROW
		IF (JRHI  .GT. IEROW) JRHI  = IEROW
		IF (ICLOW .LT. ISCOL) ICLOW = ISCOL
		IF (ICHI  .GT. IECOL) ICHI  = IECOL

C               Integrate the probability distribution over this box...

		IF (HAVEDAOPSF) TOTVAL = 10**(-0.4*(PSFMAG-25))
		IF (HAVEPSFLIB) TOTVAL = PSFTOT
		IF (HAVEDAOPSF .OR. HAVEPSFLIB) RTEMP = RNCNTS / TOTVAL
		IF (HAVEPSFLIB) THEN
		  DVDX = 1.E36
		  ICOL = NSIDE/2 + 1 - INT(COL)
		  IROW = NSIDE/2 + 1 - INT(ROW)
		END IF
                IF (HAVEPSFLIB .AND. PSFGRID) THEN
                  ixpsf = int((col-x0psf)/dxpsf)
                  iypsf = int((row-y0psf)/dypsf)
                  if (ixpsf .lt. 0 .or. iypsf .lt. 0 .or.
     &                ixpsf+1 .ge. sqrt(float(nall)) .or.
     &                iypsf+1 .ge. sqrt(float(nall))) then
                        print *, 'PSF out of interpolated bound!!!'
			print *, 'Using closest possible...could be bad!'
                        if (ixpsf .lt. 0) ixpsf = 0
                        if (iypsf .lt. 0) iypsf = 0
                        if (ixpsf+1 .ge. sqrt(float(nall))) 
     &                         ixpsf = sqrt(float(nall)) - 2
                        if (iypsf+1 .ge. sqrt(float(nall))) 
     &                         iypsf = sqrt(float(nall)) - 2
                  end if
                  xxx = (col - (x0psf+ixpsf*dxpsf))/dxpsf
                  yyy = (row - (y0psf+iypsf*dypsf))/dypsf
                  ix = int(xxx)
                  iy = int(yyy)
                  ww1 = iy+1.-yyy
                  ww2 = ix+1.-xxx
                  ww3 = yyy-iy
                  ww4 = xxx-ix
                  w1 = ww1 * ww2
                  w2 = ww3 * ww2
                  w3 = ww3 * ww4
                  w4 = ww1 * ww4
                END IF

		DO 8704 JJ=JRLOW,JRHI
		  DO 8705 II=ICLOW,ICHI

		    IF (HAVEDAOPSF) THEN
			XX = FLOAT(II) - COL
			YY = FLOAT(JJ) - ROW
			QVAL=VALUE(XX,YY,GAUSS, PSF, NPSF, DVDX,
     &                           DVDY)
			HERE = QVAL * RTEMP
		    ELSE IF (HAVEPSFLIB .AND. 
     &                   .NOT. PSFALL .AND. .NOT. PSFGRID) THEN
			call cclibget(location,npsflib,nlib,ii,jj,
     &                          col,row,qval,dvdx,dvdy)
			HERE = QVAL * RTEMP
		    ELSE IF (HAVEPSFLIB .AND. PSFALL) THEN
			call ccliball(location,npsflib,nlib,ii,jj,
     &                     col,row,allval,alldvdx,alldvdx,nall,nbytes)
			CALL GPCOORDS (ROW,COL,XXX,YYY)
			CALL GPSFVAL(XXX,YYY,ALLVAL,QVAL)
			HERE = QVAL * RTEMP
                    ELSE IF (HAVEPSFLIB .AND. PSFGRID) THEN
#ifdef NOTDEF
                        do iix=0,1
                          do iiy=0,1
                            noff = ixpsf+iix + (iypsf+iiy)*sqrt(float(nall))
                            call cclibone(location,npsflib,nlib,ii,jj,col,row,
     &                psfval(iix,iiy),psfdvdx(iix,iiy),psfdvdy(iix,iiy),
     &                nall,nbytes,nsamp,noff,nskip)
                          end do
                        end do
                        val = bin(xxx+1,yyy+1,psfval,2,2,.false.)
C	                oldval = val
#endif
                      call cclibnew(location,npsflib,nlib,ii,jj,col,row,
     &                     psfval,psfdvdx,psfdvdy,
     &                     nall,nbytes,nsamp,ixpsf,iypsf,nskip,0,ierr)
                      val = psfval(0,0)*w1 + psfval(0,1)*w2 +
     &                     psfval(1,1)*w3 + psfval(1,0)*w4
                      here = val * rtemp

		    ELSE

			PROB = 0.0

C                       We divide each pixel up into NDIV**2 segments
C                       so that the rapidly-varying probability
C                       function near the star center can be properly
C                       integrated.

			IF ( ((ROW-JJ)**2 + (COL-II)**2)**0.5 .LT. 3)
     &                    THEN
				NDIV = NDIV1
			ELSE
				NDIV = NDIV2
			END IF

			DELTA = 1.0 / NDIV
			DROW = -(NDIV/2)/FLOAT(NDIV)

			DO 8706 JJJ=1,NDIV

				ROWHERE = FLOAT(JJ) + DROW
				ROWDIFF = ROWHERE - ROW
				DCOL = -(NDIV/2)/FLOAT(NDIV)

				DO 8707 III=1,NDIV
					COLHERE = FLOAT(II) + DCOL
					COLDIFF = COLHERE - COL

C                                       Compute the distance in pixels
C                                       from the current location to
C                                       center of the star.

					DIST  = ROWDIFF**2
     &                                        + COLDIFF**2
					DIST  = SQRT(DIST)

C                                       Find the probability function
C                                       at this distance.

					NBIN  = 1
     &                                        + NINT(DIST / BINCONV)
					NBIN = MIN(NBIN,NTABLE)
					PROBHERE = STABLE(NBIN)

C                                       We also have to multiply the
C                                       probability by the area in the
C                                       pixel.

					PROBHERE=PROBHERE/(NDIV**2)

C                                       Increment the probability
C                                       for this particular pixel.

					PROB = PROB + PROBHERE

					DCOL = DCOL + DELTA
8707                            CONTINUE
				DROW = DROW + DELTA
8706                    CONTINUE

C                       Figure the number of photons that will
C                       go here.

			HERE = PROB * RNCNTS

		    END IF

C                   Add these photons to the star.

		    A(II,JJ) = A(II,JJ) + HERE
		    SUMHERE = SUMHERE + HERE
		    IF (TRUNC .AND. .NOT. PHOTNOISE 
     &                              .AND. RN .EQ. 0.) THEN
			A(II,JJ) = FLOAT ( INT(A(II,JJ)) )
			SUMHERE = FLOAT ( INT(SUMHERE) )
		    END IF
8705              CONTINUE
8704            CONTINUE

C               Store the position in a photometry file if we are to do so.

		IF (HAVEPHOT) THEN
			NSTARS               = NSTARS + 1
			PHOTDATA(ROWLOC)    = ROW
			PHOTDATA(COLLOC)    = COL
			PHOTDATA(ARTCOUNT) = RNCNTS
			PHOTDATA(BACKND)    = AVERAGE
C			ITIME=INHEAD('EXPOSURE',HEADBUF(1,IM))
			CALL CCINHEAD('EXPOSURE',HEADBUF(1,IM),ITIME)
			PHOTDATA(EXPOSURETIME) = FLOAT(ITIME)
			PHOTDATA(FRAMERA) = RA
			PHOTDATA(FRAMEDEC) = DEC
			PHOTDATA(HOURANGLE) = HA
			PHOTDATA(AIRMASS) = AM
			HJD = MAX(HJD,2444000.D0)
			HJD = MIN(HJD,2450000.D0)
                        PHOTDATA(JULIAN)  = SNGL(HJD-2444000.D0)
C			DVAL = FHEAD('GAIN',HEADBUF(1,IM))
			CALL CCFHEAD('GAIN',HEADBUF(1,IM),DVAL)
			PHOTDATA(PHOTGAIN) = DVAL
			IF (DVAL .EQ. 0) PHOTDATA(PHOTGAIN) = 1.
C			DVAL = FHEAD('RONOISE',HEADBUF(1,IM))
			CALL CCFHEAD('RONOISE',HEADBUF(1,IM),DVAL)
			PHOTDATA(PHOTRN) = DVAL
			PHOTDATA(UTMONTH)      = MONTH
			PHOTDATA(UTDAY)        = DAY
			PHOTDATA(UTYEAR)       = 99
			PHOTDATA(UTEXPOS)      = UT * 3600.

			CALL PUTPHOTREC(PHOTLABEL,PHOTDATA,NSTARS)
			IF (XERR) RETURN
		END IF

C               Print the location of the star and the brightness.

	     IF (.NOT. SILENT) WRITE(olu,*) ROW, COL, RNCNTS, SUMHERE
8703    CONTINUE

	IF (HAVESTR) GOTO 2001

 2999   CONTINUE
	IF (HAVESTR) CLOSE(3)


C   Add the sky background and do the photon noise, taking
C      into account the gain conversion
	
	DO 8708 J=ISROW, IEROW
	   DO 8709 K=ISCOL, IECOL
	      A(K,J) = A(K,J) + AVERAGE  
	      IF ( (A(K,J) .GT. 0.) .AND. (PHOTNOISE) ) THEN
		 IF (POISS) THEN
C       Use Poisson not Gaussian (although Poisson = Gaussin for large X0
		    A(K,J) = POISSON(A(K,J)*CONVERSN,KERNEL)/CONVERSN
		 ELSE IF (GALAXY) THEN
		    
		    A(K,J) = A(K,J) - AVERAGE
		    IAVE = 0
		    
		    IF(A(K,J) .GT. 0) THEN
		       DO 9091 z=1,3
			  SBMAG = -2.5*ALOG(A(K,J)*CONVERSN) + 25
			  SBERR = 0.000001*EXP((SBMAG-4.0)/1.89) + 0.005
			  IERR = A(K,J)*(10**(0.4*SBERR) - 1)*GASDEV(KERNEL)
			  IAVE = IAVE + (A(K,J) + IERR)*CONVERSN
 9091		       CONTINUE 
		    END IF
		    
		    A(K,J) = IAVE/(3*CONVERSN) + AVERAGE
		 ELSE
		    A(K,J) = A(K,J) +  SQRT(A(K,J)/CONVERSN)*GASDEV(KERNEL)
		 END IF

		 IF (TRUNC .AND. (RN .EQ. 0.0)) THEN
		    A(K,J) = FLOAT ( INT(A(K,J)) )
		 END IF 
	      END IF
 8709	   CONTINUE
 8708	CONTINUE
	
C       If the readout noise is 0.0, end.

	IF (RN .EQ. 0.0) GOTO 9999

	SIGMA = RN / CONVERSN

C       For each pixel in the image ...

	DO 8710 J=ISROW,IEROW
	    DO 8711 I=ISCOL,IECOL
		A(I,J) = SIGMA * GASDEV(KERNEL) + A(I,J)
		IF (TRUNC) A(I,J) = FLOAT ( INT(A(I,J)) )
8711        CONTINUE
8710    CONTINUE

9999    CONTINUE

	CLOSE(22)

	IF (HAVEPSFLIB) THEN
	   CALL CCFREE(NBYTES*NALL,LOCATION)
	END IF

	RETURN

575     PRINT *, 'Error reading PSFLIB file'
	XERR = .TRUE.
	RETURN
	END

	SUBROUTINE LIBREAD(LIBDAT,NSIDE,MAXLIB,NPSF,IPSF)

	INTEGER*2 LIBDAT(NSIDE*NSIDE,MAXLIB,NPSF)

	READ(21)
	READ(21)
	READ(21) (LIBDAT(I,1,IPSF),I=1,NSIDE*NSIDE*MAXLIB)

	RETURN
	END


C  Generate a Poisson distribution for mean of X
        FUNCTION POISSON(X,KERNEL)

        REAL POISSON, X , MU
        INTEGER KERNEL
        LOGICAL INIT(30)
        REAL*8 FACT, P(0:50), PINT(0:50,30)
        DATA INIT/30*.FALSE./

	POISSON = 0.

C  If x>100, treat this as Gaussian
        IF (X .GT. 30) THEN
          POISSON = X + GASDEV(KERNEL)*SQRT(X)
        ELSE

C  Otherwise we''ll tabulate the integrated probability function and
C    choose from it at random. We''ll only do this for the nearest
C    integers, however. We''ll just sample the probability function
          MU = FLOAT(NINT(X))
          IX = NINT(X)
          IF (IX .EQ. 0) THEN
            RETURN
          END IF

          IF (.NOT. INIT(IX)) THEN
            DO I = 0, 50
              IF (I .EQ. 0) THEN
                P(I) = EXP(-MU)
              ELSE
C  Avoid overflows on: P(I) = MU**I * EXP(-MU) / FACT(I)
                J = I
                P(I) = MU
                DO II = 1, I-1
                  P(I) = P(I) * MU / J
                  J = J - 1
                END DO
                P(I) = P(I) * EXP(-MU)
              END IF
            END DO
            DO I = 0, 50
              PINT(I,IX) = 0.
            END DO
C  Compute normalized integral probability
            DO I = 0, 50
              DO J = 0, I
                PINT(I,IX) = PINT(I,IX) + P(J)
              END DO
            END DO
            DO I = 0, 50
              PINT(I,IX) = PINT(I,IX) / PINT(50,IX)
            END DO
            INIT(IX) = .TRUE.

          END IF

C  Choose a random deviate
          MU = RAN1(KERNEL)
          DO I = 0, 50
            IF (PINT(I,IX) .GT. MU) THEN
              POISSON = I
              RETURN
            END IF
          END DO
        END IF

        RETURN
        END

C       Function that computes the surface brightness error given a 
C       a surface brightness
	
	FUNCTION SBERR(sbmag)
	real*4 SBERR
	real*4 sbmag
	
	SBERR=0.000001*exp((sbmag-4.0)/1.89) + 0.005

	return 
	end
