#include "Vista.h"
      SUBROUTINE ALIGN(A,ISRA,IERA,ISCA,IECA)

C     Row-by-Row Re-binning or Re-sampling of an image.
C     (In particualr Wavelength scale linearization and conversion)

C     This routine will convert each row of an image on any pixel-scale
C     (wavelength in particular) to either a linear or a logarithmic scale as
C     determined by a previous calibration or as specified in the keywords. If
C     the image is raw, with a polynomial wavelength scale, the intensities
C     may also be reduced to an uncalibrated F-lambda flux scale (i.e.,
C     rescaled to intensity units per second of time per angstrom). The notes
C     refer to angstroms, but any units apply (pixels, time, ra, etc).
C
C     Keywords:
C         [none]         Default conversion of a pixel scale into a rounded
C                        linear lambda scale.
C         DSP=f          Convert to dispersion 'f' in angstroms/pixel
C         LOG            Put image rows on a logarithmic scale such that
C                        Ln(lambda) = a + b*p, where p is the column
C                        number. In this case the DSP=f specifies the
C                        dispersion of the first column or at the wavelength
C                        specified by the W=l,p keyword.
C         W=l,p          Set lambda 'l' to occur at pixel 'p'
C         NW=N           Total number of pixels in the output (default=maximuum)
C         MS=n           Adjust to scale of image in buffer 'n'
C         FLIP           Flip w/r dispersion (flip columns before aligning).
C         V=f            Remove velocity shift 'f' in km/s
C         Z=f, DERED=f   Remove redshift 'f' (i.e. deredshift by 'f').
C         RED=f          Redshift the image rows by 'f'.
C         DP=dp & DS=ds  To remove shifts and stretches with respect the scale
C                        given by the FITS cards. Removes a shift of dp pixels
C                        and/or a pixel-stretch of a factor ds from each row
C                        of the image with respect the FITS calibration. The
C                        shift and stretching are applied using the linear
C                        transformations given by:
C                            PIXEL(image) = PIXEL(fits)*DS + DP, or
C                            PIXEL(fits) = (PIXEL(image)-DP)/DS.
C         LCMAP=b        To remove more general distortions (with respect to the
C                        scale on the FITS header) than the linear distortion
C                        case allowed by keywords DP= and DS=, or than the row-by-
C                        row polynomic case (LCMOD). LCMAP= reads a map of a 
C                        distortion for each row. Buffer b contains the map
C                          PIX(ROW,COL(IMAGE)) = B(ROW,COLUMN(FITS))
C         LCMOD=b        Like LCMAP, butwhen the distortion-map is polynomic,
C                        removing a different polynomial distortion from each row.
C                        Buffer b must contain the polynomial coefficients for
C                        each row (in the format generated by command ROWFIT).
C                            PIXEL(image) = POLY(PIXEL(fits))
C         NOFLUX         If data is in polynomial scale, inhibits conversion
C                        to flux/sec/Ang. When rebining, preserves total flux
C                        rather than flux/pixel.
C         SILENT         Suppresses printout.
C         LGI            Use lagrangian-interpolation to RE-BIN the data.
C         SINC           Use SINC-interpolation to RE-SAMPLE the data.

C       Author: J.Jesus Gonzalez        Nov, 1987
C               Lick Observatory
C               University of California
C               Santa Cruz, CA 95064
C       Modified:
C       Oct 1995 J.Jesus Gonzalez: added LCMAP, double-precision, code optimization.

#ifdef VMS
      INCLUDE 'VINCLUDE:VISTALINK.INC'       ! Communication with VISTA
      INCLUDE 'VINCLUDE:IMAGELINK.INC'       ! Image parameters
      INCLUDE 'VINCLUDE:WORK.INC'
#else
      INCLUDE 'vistadisk/source/include/vistalink.inc'
      INCLUDE 'vistadisk/source/include/imagelink.inc'
      INCLUDE 'vistadisk/source/include/work.inc'
#endif

      REAL A(IECA-ISCA+1,ISRA:IERA), BIN(3,(WORKSIZE-21)/3)
      REAL G(2), F, C, CEN
      DOUBLE PRECISION WV0, WV1, W1, W2, WVC, ZP1, WMX, TMP
      DOUBLE PRECISION LAMPIX, PIXLAM
      CHARACTER PARM*8, CARD*70, FSTRCAT*80
      LOGICAL CHANGE, FLIP, POLYN, NEWLINEAR, SINC, DOFLUX, TRACE
      LOGICAL HAVEZERO, HAVEDISP, MATCH, SILENT, KEYCHECK, ANGS, LCMAP
      COMMON DOFLUX
      COMMON POLYN
      COMMON LCMAP
      COMMON /WORK/ WV0,WV1,W2,WVC,ZP1,WMX,TMP,C,CEN,F,G,DP,DS,BIN

C     Exit if we are given keywords we do not understand.
      CALL KEYINIT
      CALL KEYDEF('DSP=')
      CALL KEYDEF('W=')
      CALL KEYDEF('NW=')
      CALL KEYDEF('LOG')
      CALL KEYDEF('FLIP')
      CALL KEYDEF('V=')
      CALL KEYDEF('Z=')
      CALL KEYDEF('DERED=')
      CALL KEYDEF('RED=')
      CALL KEYDEF('MS=')
      CALL KEYDEF('DP=')
      CALL KEYDEF('DS=')
      CALL KEYDEF('LCMAP=')
      CALL KEYDEF('LCMOD=')
      CALL KEYDEF('SILENT')
      CALL KEYDEF('NOFLUX')
      CALL KEYDEF('LGI')
      CALL KEYDEF('SINC')
      CALL KEYDEF('TRACE')

      IF (.NOT. KEYCHECK()) THEN
	  XERR = .TRUE.
	  RETURN
      END IF

C     Initialize parameters:
C     Redshift plus one, Pixel shift and scale factor.
      ZP1     = 1.0D0
      DP      = 0.0E0
      DS      = 1.0E0
C     Flip dispersion order?
      FLIP    = .FALSE.
C     Change dispersion zero point? Original Image on polyn. scale?
      CHANGE  = .FALSE.
      POLYN   = .FALSE.
C     Convert to FN/sec/Ang. Assume we want linear scale
      DOFLUX  = .TRUE.
      NEWLINEAR = .TRUE.
C     Zero point defined with W=()? Dispersion defined with DSP= ?
      HAVEZERO = .FALSE.
      HAVEDISP = .FALSE.
C     Match scale of another image? Don''t print results.
      MATCH   = .FALSE.
      SILENT  = .FALSE.
C     No line-curvature model. Re-Binning with lagrangian-interpolation.
      LCMAP = .FALSE.
      MOD = 0
      SINC = .FALSE.
      TRACE = .FALSE.
      NCOLN = 0

C     Check for keywords to overide the default dispersions.
      DO 8701 I=1, NWORD
C         New zero point
	  IF (WORD(I)(1:2) .EQ. 'W=') THEN
	      CALL ASSIGNV(WORD(I),2,G,NG,PARM)
	      IF (XERR) RETURN
	      IF (NG.LT.2) THEN
		  PRINT *, 'Must specify both wavelength and',
     &		' pixel number with W=keyword'
		  XERR = .TRUE.
		  RETURN
	      END IF
	      CHANGE = .TRUE.
	      HAVEZERO = .TRUE.
	      WVC = G(1)
	      CEN = G(2)

	  ELSE IF (WORD(I)(1:3) .EQ. 'NW=') THEN
C         Set the number of columns (wavlengths) on the output image.
	      CALL ASSIGN(WORD(I),F,PARM)
              NCOLN = NINT(F)
              IF (NCOLN.GT.(WORKSIZE-21)/3) THEN
                  PRINT*,'NW limited to ',(WORKSIZE-21)/3
                  XERR = .TRUE.
                  RETURN
              END IF
	      CHANGE = .TRUE.

	  ELSE IF (WORD(I)(1:3) .EQ. 'LGI') THEN
C         Re-bin using lagrangian interpolation.
	      SINC = .FALSE.

	  ELSE IF (WORD(I)(1:4) .EQ. 'SINC') THEN
C         Re-sample using Sinc interpolation.
	      SINC = .TRUE.

	  ELSE IF (WORD(I)(1:3) .EQ. 'DP=') THEN
C         Lambda-shift
	      CALL ASSIGN(WORD(I),DP,PARM)
	      IF (XERR) RETURN

	  ELSE IF (WORD(I)(1:3) .EQ. 'DS=') THEN
C         Scale change.
	      CALL ASSIGN(WORD(I),DS,PARM)
	      IF (XERR) RETURN

	  ELSE IF (WORD(I)(1:2) .EQ. 'V=') THEN
C         Radial-Velocity
	      CALL ASSIGN(WORD(I),F,PARM)
	      IF (XERR) RETURN
	      ZP1=DSQRT((2.997925D5+DBLE(F))/
     &	              (2.997925D5-DBLE(F)))

	  ELSE IF (WORD(I)(1:2) .EQ. 'Z=' .OR.
C         De-redshift
     &        WORD(I)(1:6) .EQ. 'DERED=') THEN
	      CALL ASSIGN(WORD(I),F,PARM)
	      IF (XERR) RETURN
	      ZP1 = DBLE(F) + 1.0D0

	  ELSE IF (WORD(I)(1:4) .EQ. 'RED=') THEN
C         Redshift.
	      CALL ASSIGN(WORD(I),F,PARM)
	      IF (XERR) RETURN
	      ZP1 = 1.D0/(1.D0 + DBLE(F))

	  ELSE IF (WORD(I)(1:4) .EQ. 'DSP=') THEN
	      CALL ASSIGN(WORD(I),F,PARM)
	      IF (XERR) RETURN
	      HAVEDISP = .TRUE.
	      WV1 = DBLE(F)

	  ELSE IF(WORD(I)(1:4) .EQ. 'LOG') THEN
	      NEWLINEAR = .FALSE.

	  ELSE IF(WORD(I)(1:4) .EQ. 'SILE') THEN
	      SILENT = .TRUE.

	  ELSE IF (WORD(I)(1:3) .EQ. 'MS=') THEN
C         Match other image
	      CALL ASSIGN(WORD(I),F,PARM)
	      IF (XERR) RETURN
	      IM1 =NINT(F)
	      IF (.NOT. BUFF(IM1)) THEN
		  PRINT *,' Matching image does not exist'
		  XERR = .TRUE.
		  RETURN
	      END IF

	      WV0 = SPWAVE(1,IM1)
	      WV1 = SPWAVE(2,IM1)
              NCOLN = ICOORD(NNCOL,IM1)
	      IF(LAMTYPE(IM1) .EQ. LAMBDALINEAR) THEN
		  NEWLINEAR = .TRUE.
	      ELSE IF(LAMTYPE(IM1) .EQ. LAMBDALOG) THEN
		  NEWLINEAR = .FALSE.
	      ELSE
                  IF (CTYPE1(IM1)(1:3).EQ.'Ln_') THEN
		      NEWLINEAR = .FALSE.
                  ELSE
	   	      NEWLINEAR = .TRUE.
                  END IF
	      END IF
	      HAVEDISP = .TRUE.
	      HAVEZERO = .TRUE.
	      MATCH = .TRUE.

	  ELSE IF (WORD(I)(1:6) .EQ. 'LCMOD=') THEN
C         Polymomial line-curvature model.
	      CALL ASSIGN(WORD(I),F,PARM)
	      IF (XERR) RETURN
	      MOD = NINT(F)
	      LCMAP = .FALSE.

	  ELSE IF (WORD(I)(1:6) .EQ. 'LCMAP=') THEN
C         line-curvature model Generic Map.
	      CALL ASSIGN(WORD(I),F,PARM)
	      IF (XERR) RETURN
	      MOD = NINT(F)
	      LCMAP = .TRUE.

	  ELSE IF (WORD(I) .EQ. 'NOFLUX') THEN
	      DOFLUX = .FALSE.

	  ELSE IF (WORD(I) .EQ. 'FLIP') THEN
C         Flip dispersion
	      FLIP = .TRUE.
	      CHANGE = .TRUE.

	  ELSE IF (WORD(I) .EQ. 'TRACE') THEN
C         Extra i/o for debugging.
	      TRACE = .TRUE.

	  END IF
8701  CONTINUE

C     See if the lambda calibration has units of angstroms in a form
C     VISTA recognizes. If this is not the case, the aligment will
C     be done using as lambda-calibration the linear pixel scale given
C     by the FITS cards CRVAL1, CDELT1 and CRPIX1.
      IF (LAMTYPE(IM) .EQ. LAMBDALINEAR .OR.
     &    LAMTYPE(IM) .EQ. LAMBDALOG .OR.
     &    LAMTYPE(IM) .EQ. LAMBDALOG10) THEN
           ANGS = .TRUE.
      ELSE IF (LAMTYPE(IM) .EQ. LAMBDAPOLY) THEN
           ANGS = .TRUE.
           POLYN = .TRUE.
      ELSE
           ANGS=.FALSE.
      END IF

C     Get the polynomial distortion map if any.
      IF (MOD .GT. 0) THEN
	  CALL GETIMDATA(LOC1,ISRM,IERM,I,ISCM,IECM,J,MOD)
	  IF (ISRA.GT.IERM .OR. IERA.LT.ISRM) THEN
	     PRINT*,' Line-Curvature Map and Image do not overlap.'
	     PRINT*,' Line-Curvature will be extrapolated with a constant.'
C	     XERR = .TRUE.
	  ELSE IF (IERA.GT.IERM .OR. ISRA.LT.ISRM) THEN
	     PRINT*,' WARNING: Line-Curvature Map does not cover all rows.'
	     PRINT*,' Line-Curvature will be extrapolated with a constant.'
	  END IF
	  IF (XERR) RETURN
C	  CALL CCMODLIM(LOC1,1,NCOL,MSC,MEC)
       END IF

C     Wavelength limits of the array. IDC relates pixels to array elements.
      IDC = ICOORD(IXSC,IM) - 1
      TMP = DBLE((ISCA-0.5-DP)/DS) - DBLE(IDC)
      W1 = LAMPIX(TMP,IM)/ZP1
      TMP = DBLE((IECA+0.5-DP)/DS) - DBLE(IDC)
      W2 = LAMPIX(TMP,IM)/ZP1
      NCOL = IECA - ISCA + 1

C     If dispersion not given, use the average dispersion.
      IF (.NOT. HAVEDISP) THEN
	  PRINT *,' No dispersion (DSP=d) given on command line'
	  WV1 = (W2-W1)/DBLE(NCOL)
	  PRINT *,' Will use a dispersion value of ',REAL(WV1)
      END IF

C     Check for dispersion flip and direction of increasing pixels.
      IF (FLIP) THEN
          WV1 = -WV1
      END IF

      IF (WV1*(W2-W1).GT.0) THEN
          ISIGN = +1
      ELSE
          ISIGN = -1
      END IF

C     Default refference to first pixel or last pixel.
      IF(.NOT. HAVEZERO) THEN
          CEN = 5.0E-1
          IF (ISIGN.GT.0) THEN
              WVC = W1
          ELSE
              WVC = W2
          END IF
	  CHANGE = .TRUE.
      END IF

C     Translate zero point of scale from that given by W= keyword
      IF (CHANGE) THEN
	  IF (MATCH) THEN
	      PRINT *,'Inconsistent keywords: Can not match',
     &	        ' another image and change parameters too.'
	      XERR = .TRUE.
	      RETURN
	  END IF
	  IF(NEWLINEAR) THEN
	      WV0 = WVC- WV1*DBLE(CEN)
	  ELSE IF (WVC.GT.0.0) THEN
	      WV1 = WV1 / WVC
	      WV0 = DLOG(WVC) - WV1*DBLE(CEN)
          ELSE
              PRINT 70
              XERR = .TRUE.
              RETURN
	  END IF
      END IF

 70   FORMAT(' Image is not well suited for a Logarithmic Scale.'
     &  ,/,' Use w=l,p to force non-negative physical coordinates',
     &  ' in the new scale.')

C     Find the size of the output image (maximum number of bins).
      IF (NCOLN.LE.0) THEN
          IF (ISIGN .GT. 0) THEN
              WMX = W2
          ELSE
              WMX = W1
          END IF
          IF(NEWLINEAR) THEN
              NCOLN = NINT(ABS((WMX-WV0-(WV1/2.))/WV1))
          ELSE IF (WMX.GT.0.0) THEN
              NCOLN = NINT(ABS((DLOG(WMX)-WV0-(WV1/2.))/WV1))
          ELSE
              PRINT 70
              XERR = .TRUE.
              RETURN
          END IF
          NCOLN = MIN(NCOLN,(WORKSIZE-21)/3)
      END IF

      IF (TRACE) THEN
      print*,'cen, wvc, w1, w2, wmx, wv0', cen, wvc, w1, w2, wmx, wv0
      w1=lampix(3.0d0,im)
      print*,'ncoln, wv(3), pix(wv(3))', ncoln, w1, pixlam(w1,im)
      END IF

C     To rebin the old image, we calculate the extreme lambdas of each
C     bin in the new image. If a velocity correction is desired, lambdas
C     are adjusted to the observed wavelength in the original image. The
C     location of these lambdas in the original pixel space is calculated
C     using the inverse dispersion. The limits and size of the new bins in
C     pixel scale are loaded into array BIN. If the lambda type is not one
C     recongnizable by VISTA, the rebining will be donne in the natural units
C     of the image (not necessaraly wavelength) but limits may need to be
C     adjusted to compensate that PIXLAM returns array elements not pixels.
      IF (ISIGN.GT.0) THEN
          IMN = 1
          IMX = 2
      ELSE
          IMN = 2
          IMX = 1
      END IF

C     Adjust DP to transform limits from array element to pixel units.
      IF (IDC.NE.0) DP = REAL(IDC)*DS + DP

      W1 = WV0 - WV1*5.0D-1
      W2 = WV0 + WV1*5.0D-1
      IF (NEWLINEAR) THEN
C         Bining into a linear scale.
          IF (DP.EQ.0. .AND. DS.EQ.1) THEN
	      DO 9702 I=1,NCOLN
	        BIN(IMN,I) = REAL(PIXLAM((W1+I*WV1)*ZP1,IM))
	        BIN(IMX,I) = REAL(PIXLAM((W2+I*WV1)*ZP1,IM))
9702          CONTINUE
          ELSE IF (DS.EQ.1) THEN
	      DO 9703 I=1,NCOLN
	        BIN(IMN,I) = REAL(PIXLAM((W1+I*WV1)*ZP1,IM)) + DP
	        BIN(IMX,I) = REAL(PIXLAM((W2+I*WV1)*ZP1,IM)) + DP
9703          CONTINUE
          ELSE IF (DP.EQ.0) THEN
	      DO 9704 I=1,NCOLN
	        BIN(IMN,I) = REAL(PIXLAM((W1+I*WV1)*ZP1,IM))*DS
	        BIN(IMX,I) = REAL(PIXLAM((W2+I*WV1)*ZP1,IM))*DS
9704          CONTINUE
          ELSE
	      DO 9705 I=1,NCOLN
	        BIN(IMN,I) = REAL(PIXLAM((W1+I*WV1)*ZP1,IM))*DS+DP
	        BIN(IMX,I) = REAL(PIXLAM((W2+I*WV1)*ZP1,IM))*DS+DP
9705          CONTINUE
          END IF
      ELSE
C         Binning into logarithmic scale.
          IF (DP.EQ.0. .AND. DS.EQ.1) THEN
	      DO 9706 I=1,NCOLN
	        BIN(IMN,I)=REAL(PIXLAM(ZP1*DEXP(W1+I*WV1),IM))
	        BIN(IMX,I)=REAL(PIXLAM(ZP1*DEXP(W2+I*WV1),IM))
9706          CONTINUE
          ELSE IF (DS.EQ.1) THEN
	      DO 9707 I=1,NCOLN
	        BIN(IMN,I)=REAL(PIXLAM(ZP1*DEXP(W1+I*WV1),IM))+DP
	        BIN(IMX,I)=REAL(PIXLAM(ZP1*DEXP(W2+I*WV1),IM))+DP
9707          CONTINUE
          ELSE IF (DP.EQ.0.0) THEN
	      DO 9708 I=1,NCOLN
	        BIN(IMN,I)=REAL(PIXLAM(ZP1*DEXP(W1+I*WV1),IM))*DS
	        BIN(IMX,I)=REAL(PIXLAM(ZP1*DEXP(W2+I*WV1),IM))*DS
9708          CONTINUE
          ELSE
	      DO 9709 I=1,NCOLN
	        BIN(IMN,I)=REAL(PIXLAM(ZP1*DEXP(W1+I*WV1),IM))*DS+DP
	        BIN(IMX,I)=REAL(PIXLAM(ZP1*DEXP(W2+I*WV1),IM))*DS+DP
9709          CONTINUE
          END IF
      END IF

C     If polynomial scale, normalize the image to f-lambda. The pixels are
C     divided by their widths in angstroms and the exposure time to yield
C     a scale of flux/(sec*angstrom). Use BIN(3,I) as normalization factor.
      IF (POLYN .AND. DOFLUX) THEN
	  CALL CCFHEAD('EXPOSURE',HEADBUF(1,IM),TMP)
	  IF (TMP.LE.0.) TMP=1.0D0

	  IF (NEWLINEAR) THEN
	      C = REAL(1.0D0/(TMP*DABS(WV1)))
	      DO 8704 I=1,NCOLN
	          BIN(3,I) = C
8704          CONTINUE
	  ELSE
	      C = REAL(1.0D0/(TMP*DABS(DEXP(W2)-DEXP(W1))))
	      DO 8705 I=1,NCOLN
	          BIN(3,I) = REAL(C*DEXP(-I*WV1))
8705          CONTINUE
	  END IF
      ELSE IF (SINC) THEN
	  DOFLUX = .FALSE.
      END IF

C     Create the output image to later load it into the original.
      TEMPHEAD = ' '
      CALL CCVSTRCPY(TEMPHEAD,HEADBUF(1,IM))

C     Remove unwanted cards from the  FITS header
      IF(POLYN) THEN
	  CALL UNFIT('LPOLY1',TEMPHEAD)
	  CALL UNFIT('IPOLY1',TEMPHEAD)
	  CALL UNFIT('LPOLY0',TEMPHEAD)
	  CALL UNFIT('IPOLY0',TEMPHEAD)
	  CALL UNFIT('LAMORD',TEMPHEAD)
      END IF

C     Set the Number of pixels, the calibration type, the Wavelength at
C     pixel CRPIX1, the Dispersion and the Pixel of wavelength CRVAL1.
      CALL INHEADSET('NAXIS1',NCOLN,TEMPHEAD)
      IF (ANGS) THEN
          W1 = (WV0+WV1) * 1.0D-10
          W2 = WV1 * 1.0D-10
          IF(NEWLINEAR) THEN
             CALL CHEADSET('CTYPE1','LAMBDA',TEMPHEAD)
          ELSE
             CALL CHEADSET('CTYPE1','LOG_LAMBDA',TEMPHEAD)
          END IF
      ELSE
	  W1 = WV0 + WV1
	  W2 = WV1
          IF (.NOT.NEWLINEAR) THEN
             IF (CTYPE1(IM).EQ.' ') CTYPE1(IM) = 'PIXEL'
             CARD = FSTRCAT('Ln_',CTYPE1(IM))
             CALL CHEADSET('CTYPE1',CARD,TEMPHEAD)
          END IF
      END IF
      CALL FHEADSET('CRVAL1',W1,TEMPHEAD)
      CALL FHEADSET('CDELT1',W2,TEMPHEAD)
      CALL INHEADSET('CRPIX1',1,TEMPHEAD)

      CALL CREATEIM(LOCIM,NSR,NER,NSC,NEC,1,TEMPHEAD,.FALSE.)
      IF (XERR) GO TO 1000

C--   Rebin the data into the new image, and release old image.
      IF (MOD .EQ. 0) THEN
          IF (SINC) THEN
            CALL CCRESAM(LOCIM,A,ISRA,IERA,ISCA,IECA,NCOLN,BIN)
	  ELSE
            CALL CCREBIN(LOCIM,A,ISRA,IERA,ISCA,IECA,NCOLN,BIN)
	  END IF
      ELSE
	  IF (SINC) THEN
	      CALL CCRESAMLC(LOCIM,A,ISRA,IERA,ISCA,IECA,NCOLN,
     &          BIN,LOC1,ISRM,IERM,ISCM,IECM,LCMAP)
	  ELSE
	      CALL CCREBINLC(LOCIM,A,ISRA,IERA,ISCA,IECA,NCOLN,
     &          BIN,LOC1,ISRM,IERM,ISCM,IECM)
	  END IF
      END IF

C     Print out new scale.
      IF (SILENT) GOTO 1000

      W1 = WV0+WV1
      W2 = WV0+NCOLN*WV1
      IF (NEWLINEAR) THEN
          WRITE(olu,'(/,A)',ERR=1000)
     &  '    Pixel Scale adjusted to a new Linear Scale:'
          IF (ANGS) THEN
              CARD = 'Wavelength [angs]'
          ELSE
              CARD = CTYPE1(IM)(1:17)
              IF (CARD.EQ.' ') CARD='Column Number'
          END IF
          WRITE(olu,101,ERR=1000) CARD(1:17),WV0,WV1,W1,W2,NCOLN
      ELSE
          WRITE(olu,'(/,A)',ERR=1000)
     &  '    Pixel Scale adjusted to a Logarithmic Scale:'
          IF (ANGS) THEN
              CARD = 'Ln(Wavelength)'
          ELSE
              CARD = CTYPE1(IM)(1:17)
              IF (CARD(1:8).EQ.'Ln_PIXEL') CARD='Ln(Column number)'
          END IF
          WRITE(olu,101,ERR=1000) CARD(1:17), WV0, WV1, DEXP(W1),
     &                            DEXP(W2), NCOLN
      END IF
101   FORMAT(1X,A17,' =',1PE13.5,SP,1PE13.5,S,' * (Pixel number)',/,
     &  1X,'Range: From',1PE13.5,' at pixel 001 To',1PE13.5,
     &  ' at pixel',I5)
C--   ---------------------------- Release the virtual memory. ----C
1000  CALL RELEASEOLDIMAGE

      RETURN
      END

      SUBROUTINE REBIN(B,A,ISR,IER,ISC,IEC,NBINS,BIN)
C     Use Gonzalez''s interpolation function which, unlike sinc-interpolation, 
C     takes care of the fact that input and output bins have a different size.

      REAL A(ISC:IEC,ISR:IER),B(NBINS,ISR:IER), BIN(3,NBINS)
      DOUBLE PRECISION S, A1, A2
      LOGICAL DOFLUX, POLYN
      COMMON DOFLUX
      COMMON POLYN

      DO 8704 IC=1,NBINS
C--   Assume A=0 two pixels beyond edges.
      XS = MAX(BIN(1,IC),REAL(ISC) - 2.5)
      XE = MIN(BIN(2,IC),REAL(IEC) + 2.5)

      IF (XS.GE.XE) THEN
          DO 8701 IR=ISR,IER
	    B(IC,IR) = 0.0E0
8701	  CONTINUE
	  GOTO 8704
      END IF

C--   Take care of inter(extra)polation near the edges.
      IS = MAX0(MIN0(NINT(XS),IEC-1),ISC+1)
      IE = MIN0(MAX0(NINT(XE),ISC+1),IEC-1)
      FS = XS - FLOAT(IS)
      FE = XE - FLOAT(IE)

      DO 8703 IR=ISR,IER
c--   (binned) Parabolic coefficients.
        A2 = ((A(IS+1,IR)+A(IS-1,IR))/2.D0 -A(IS,IR))/3.D0
        A1 = (A(IS+1,IR)-A(IS-1,IR))/4.D0

        IF (IS.LT.IE) THEN
c--     Left-pixel fractional contribution to Flux.
          S = (0.5D0-FS)*(A(IS,IR)+(FS+0.5D0)*(A1+A2*FS))
c--       Whole pixel contribution to Flux.
          DO 8702 I = IS+1,IE-1,1
            S = S + A(I,IR)
8702      CONTINUE
c--       Right-pixel fractional contribution to Flux
          A2 = ((A(IE+1,IR)+A(IE-1,IR))/2.D0 -A(IE,IR))/3.D0
          A1 = (A(IE+1,IR)-A(IE-1,IR))/4.D0
          S = S+(0.5D0+FE)*(A(IE,IR)+(FE-0.5D0)*(A1+A2*FE))
        ELSE
          S = A(IS,IR) - A2/4.0D0
          S = (FE-FS)*(S+(FE+FS)*A1+A2*(FS*FS+FS*FE+FE*FE))
        END IF
        B(IC,IR) = S
8703	CONTINUE
8704  CONTINUE
C--   Normalize in case of Fluxing.
      IF (DOFLUX) THEN
        IF (POLYN) THEN
          DO 8706 IR=ISR,IER
          DO 8705 IC=1,NBINS
            B(IC,IR) = B(IC,IR)*BIN(3,IC)
8705      CONTINUE
8706      CONTINUE
        ELSE
          DO 8708 IR=ISR,IER
          DO 8707 IC=1,NBINS
            B(IC,IR) = B(IC,IR)/(BIN(2,IC)-BIN(1,IC))
8707      CONTINUE
8708      CONTINUE
        END IF
      END IF

      RETURN
      END

      REAL*4 FUNCTION BINJG(A,ISR,IER,ISC,IEC,IR,XS,XE)
C     Whole pixels lying inside the new bins are directly added. Any fractional
C     pixel contribution to the bin is estimated by integration of the parabola
C     whose area matches the intensity of the pixel and its two neighbors
C     (Gonzalez-integration). NOTE: this is different from Sympson-integration
C     (good for sampled functions, but not for binned and sampled functions).
C     The Sympson-parabola C0+C1*X+C2*X*X through (-1,Ym),(0,Y0),(1,Yp), has
C     coefficients C0=Y0, C1=(Yp-Ym)/2, and C2=(Yp+Ym)/2-Y0, while the Gonzalez
C     parabola A0+A1*X+A2*X*X, has A0=C0-C2/12, A1=C1, and A2=C2. This scheme
C     takes care of the fact that the original spectrum is also binned data.

      REAL A(ISC:IEC,ISR:IER)
      DOUBLE PRECISION S, A1, A2

C--   Assume A=0 two pixels beyond edges.
      XS = MAX(XS,REAL(ISC) - 2.5)
      XE = MIN(XE,REAL(IEC) + 2.5)
      IF (XS.GE.XE) THEN
        BINJG = 0.0E0
        RETURN
      END IF

C--   Take care of inter(extra)polation near the edges.
      IS = MAX0(MIN0(NINT(XS),IEC-1),ISC+1)
      IE = MIN0(MAX0(NINT(XE),ISC+1),IEC-1)
      FS = XS - FLOAT(IS)
      FE = XE - FLOAT(IE)

c--   (binned) Parabolic coefficients.
      A2 = ((A(IS+1,IR)+A(IS-1,IR))/2.D0 -A(IS,IR))/3.D0
      A1 = (A(IS+1,IR)-A(IS-1,IR))/4.D0

      IF (IS.LT.IE) THEN
c--     Left-pixel fractional contribution to Flux.
        S = (0.5D0-FS)*(A(IS,IR)+(FS+0.5D0)*(A1+A2*FS))
c--     Whole pixel contribution to Flux.
        DO 8701 I = IS+1,IE-1,1
         S = S + A(I,IR)
8701    CONTINUE
c--     Right-pixel fractional contribution to Flux
        A2 = ((A(IE+1,IR)+A(IE-1,IR))/2.D0 -A(IE,IR))/3.D0
        A1 = (A(IE+1,IR)-A(IE-1,IR))/4.D0
        S = S+(0.5D0+FE)*(A(IE,IR)+(FE-0.5D0)*(A1+A2*FE))
      ELSE
        S = A(IS,IR) - A2/4.0D0
        S = (FE-FS)*(S+(FE+FS)*A1+A2*(FS*FS+FS*FE+FE*FE))
      END IF
      BINJG = S

      RETURN
      END

      SUBROUTINE REBINLC(B,A,ISR,IER,ISC,IEC,NBINS,BIN,
     &                   M,MSR,MER,MSC,MEC)
C     Same as REBIN but allows a distortion model in pixel space.

      REAL A(ISC:IEC,ISR:IER), B(NBINS,ISR:IER), M(MSC:MEC,MSR:MER)
      REAL BIN(3,NBINS)
      LOGICAL DOFLUX, POLYN, LCMAP
      COMMON DOFLUX
      COMMON POLYN
      COMMON LCMAP

C--   The Map is non-polynomial (more general case).

      IF (LCMAP) THEN
        PRINT *,'Re-Binning with Distortion Map'
        DO 8703 IR=ISR,IER
C--     Extrapolate LC-Map with a constant.
        IRM = MIN(MER,MAX(IR,MSR))
        DO 8702 IC=1,NBINS
C--       Get bin-limits in original image interpolating the LC-Map
C--       X = MAX(MIN(BIN(1,IC),FLOAT(MEC+2)),FLOAT(MSC-2))
          I = MAX0(MIN0(NINT(BIN(1,IC)),MEC-1),MSC+1)
          F = BIN(1,IC) - FLOAT(I)
          XS = M(I,IRM) + F*((M(I+1,IRM)-M(I-1,IRM))/2. +
     &         F*((M(I+1,IRM)+M(I-1,IRM))/2. - M(I,IRM)))

          I = MAX0(MIN0(NINT(BIN(2,IC)),MEC-1),MSC+1)
          F = BIN(2,IC) - FLOAT(I)
          XE = M(I,IRM) + F*((M(I+1,IRM)-M(I-1,IRM))/2. +
     &         F*((M(I+1,IRM)+M(I-1,IRM))/2. - M(I,IRM)))

          B(IC,IR) = BINJG(A,ISR,IER,ISC,IEC,IR,XS,XE)
          IF (DOFLUX) THEN
            IF (POLYN) THEN
               B(IC,IR) = B(IC,IR)*BIN(3,IC)
            ELSE
               B(IC,IR) = B(IC,IR)/(XE-XS)
            END IF
          END IF
8702    CONTINUE
8703    CONTINUE

      ELSE
        PRINT *,'Re-Binning with Polynomial Distortion Map'
C--     Polynomial Distortion; Order of the Line-Curvature Model.
        MO = MEC-1

        DO 8706 IR=ISR,IER
C--     Extrapolate polynomial map with a constant.
        IRM = MIN(MER,MAX(IR,MSR))

        DO 8705 IC=1,NBINS
C--       Find the bin limits in the original image.
          XS = M(MEC,IRM)
          XE = XS
          DO 8704 N = MO,1,-1
            XS = XS*BIN(1,IC) + M(N,IRM)
            XE = XE*BIN(2,IC) + M(N,IRM)
8704      CONTINUE
          B(IC,IR) = BINJG(A,ISR,IER,ISC,IEC,IR,XS,XE)
          IF (DOFLUX) THEN
            IF (POLYN) THEN
               B(IC,IR) = B(IC,IR)*BIN(3,IC)
            ELSE
               B(IC,IR) = B(IC,IR)/(XE-XS)
            END IF
          END IF
8705    CONTINUE
8706    CONTINUE
      END IF
      RETURN
      END

      SUBROUTINE RESAM(B,A,ISR,IER,ISC,IEC,NBINS,BIN)
C     The original VISTA version 2 subroutine did not rebin the data. This can
C     be very dangerous in data with badly variable dispersion, when logarithmic
C     lambda-scale is desired, or when we want to ALIGN the data with a DSP not
C     very close to the original dispersion. Sinc interpolation is used to
C     re-sample (not re-bin) the old image at the pixel corresponding to the
C     central wavelength of each new pixel.

      REAL A(ISC:IEC,ISR:IER), B(NBINS,ISR:IER), BIN(3,NBINS)
      LOGICAL DOFLUX
      COMMON DOFLUX

C--   Find central array element. SINCGEN works on array elements, not pixels.
      C = REAL(ISC-1)
      DO 9701 IC=1, NBINS
          BIN(1,IC) = (BIN(1,IC)+BIN(2,IC))/2.0 - C
 9701 CONTINUE

      NCOL = IEC-ISC+1
      DO 8703 IR=ISR,IER
      DO 8702 IC=1,NBINS
	  CALL SINCGEN(A(ISC,IR),NCOL,BIN(1,IC),B(IC,IR))
8702  CONTINUE
8703  CONTINUE

      IF (.NOT. DOFLUX) RETURN

      DO 8704 IR=ISR,IER
      DO 8705 IC=1,NBINS
	  B(IC,IR) = B(IC,IR)*BIN(3,IC)
8705  CONTINUE
8704  CONTINUE
      RETURN
      END

      SUBROUTINE RESAMLC(B,A,ISR,IER,ISC,IEC,NBINS,BIN,
     &                   M,MSR,MER,MSC,MEC,LCMAP)
C     Same as RESAM but allows a distortion model in pixel space.

      REAL A(ISC:IEC,ISR:IER), B(NBINS,ISR:IER), M(MSC:MEC,MSR:MER)
      REAL BIN(3,NBINS)
      LOGICAL DOFLUX, LCMAP
      COMMON DOFLUX

C--   SINCGEN works on array elements, not pixels.
      NCOL = IEC - ISC + 1
      C = REAL(ISC-1)

      IF (LCMAP) THEN
        PRINT *,'Resampling with Distortion Map'
        DO 8702 IR=ISR,IER
C--     Extrapolate Line-Curvature with a constant, add linear distortion.
        IRM = MIN(MER,MAX(IR,MSR))
        DO 8701 IC=1,NBINS
C--       Find the bin center in the original image.
          BC = (BIN(1,IC)+BIN(2,IC))/2.0
          I = MAX0(MIN0(NINT(BC),MEC-1),MSC+1)
          F = BC - FLOAT(I)
          XTERP = M(I,IRM)+F*((M(I+1,IRM)-M(I-1,IRM))/2. +
     &            F*((M(I+1,IRM)+M(I-1,IRM))/2. - M(I,IRM)))
          CALL SINCGEN(A(ISC,IR),NCOL,XTERP-C,B(IC,IR))
8701    CONTINUE
8702    CONTINUE

      ELSE
        PRINT *,'Resampling with Polynomial Distortion Map'
        DO 8705 IR=ISR,IER
C--     Extrapolate Line-Curvature with a constant, add linear distortion.
        IRM = MIN(MER,MAX(IR,MSR))
        DO 8704 IC=1,NBINS
C--       Find the bin center in the original image.
          BC = (BIN(1,IC)+BIN(2,IC))/2.0
C--       Polynomial Order of the Line-Curvature Model.
          MO = MEC-1
          XTERP = M(MEC,IRM)
          DO 8703 N = MO,1,-1
            XTERP = XTERP*BC + M(N,IRM)
8703      CONTINUE
          CALL SINCGEN(A(ISC,IR),NCOL,XTERP-C,B(IC,IR))
8704    CONTINUE
8705    CONTINUE
      END IF

      IF (.NOT. DOFLUX) RETURN

      DO 8707 IR=ISR,IER
      DO 8706 IC=1,NBINS
        B(IC,IR) = B(IC,IR)*BIN(3,IC)
8706  CONTINUE
8707  CONTINUE

      RETURN
      END
