#include "Vista.h"
 
      SUBROUTINE IMPOST (A,ISROW,IEROW,ISCOL,IECOL)
 
C
C  The Command Formerly Known As POSTIT - Version 3.0 (1998 Feb 27)
C
C  Image Hardcopy using PostScript
C
C  This program produces a hardcopy of an image by representing intensities
C  as an 8-bit halftone gray scale.  The output is a PostScript file which
C  may be sent to any standard PostScript hardcopy device (or viewed on any
C  non-hardcopy device, such as a NeWS Window System or a Macintosh).
C
C  We work with the LickMongo PostScript graphics drivers to act as a kind 
C  of PostScript pre-processor to create a text file containing the necessary 
C  hex map of the image intensiites and PostScript graphics commands.  A 
C  complete description of the PostScript commands sent may be found in 
C  'PostScript Language Program Design', published by Adobe Systems 
C  Incorporated.
C
C  The image intensities are mapped into [0,255] in one of 2 ways:
C
C     1)  Linear mapping using a zero point and intensity span supplied by
C         the user.  The mapping is of the form:
C
C            INTENS = (IMG - ZERO)*(NLEV/SPAN)
C
C     2)  Histogram equalization.  Computes a histogram of all image pixels
C         within the region of interest, and loads the pixels into [0,NLEV-1]
C         such the image is displayed with an equal number of pixels at
C         all display intensity levels.  There are options to do this over 
C         logarithmic or square-rootintervals in addition to the 
C         default "flat" equalization.
C
C  The command structure is essentially the same as that for image display 
C  with the TV command.
C
C  The default display is black on white (like a POSS plate).  White on
C  black may be specified via a keyword, but the results are usually less than
C  satisfactory with the current generation of laser printers.
C
C  Using an optional keyword "FOUR", the default grayscale may be changed
C  to a 4 bit (16 level) grayscale.
C
C  PostScript instructions generated by this program are may be used to
C  produce the image hardcopy on any PostScript device via an appropriate
C  server.  PostScript files produced conform to the `Encapsulated
C  PostScript` structuring conventions described `PostScript Language Program 
C  Design` Manual (Adobe Systems Incorporated 1988) following the Version 2.0
C  EPS Specifications (Adobe Systems Inc. January 1989).  It should thus be 
C  possible to make images on any PostScript device that conforms to the 
C  "standard".  Images created with IMPOST have been displayed successfully on
C  the X11 window system, on displayed PostScript devices, and transmitted 
C  electronically via both email and tcp/ip with no difficulties.
C
C   'PostScript' is a registered trademark of Adobe Systems Incorporated.
C   'Macintosh' is a trademark of Apple Computers, Inc.
C   'X11' is a trademark of somebody
C
C  -----------------------------------------------
C
C  IMPOST Command Syntax:
C
C     IMPOST imbuf [BOX=b] [Z=zero] [L=span] [CLIP|NOCLIP] [POSITIVE] [TITLE]
C                  [HIST=xxx] [LAND] [FOUR] [AXES] [COMMENT] [COMMENT=xxx]
C		   [FILE=xxx] [OUT=xxx] [SCALE=s] [CEN=r,c] [FLIP] [BAR=xxx]
C                  [NOBAR] [WIND=w,h] [ORIGIN=x,y] [PAGE=L,S] [COPIES=n] 
C                  [LARGE] [INFO] [BLACK=n] [EPS] [NOBOX]
C
C   Where:
C
C     imbuf           is the buffer containing the image to be hardcopied.
C
C     BOX=b           only display the region within the given box.
C
C     Z=zero          [optional] Zero point of the intensity mapping.
C                       Default value is 0.0
C
C     L=span          [optional] Span of the intensity mapping.  If none is
C                       is specified, the default value will be taken to be
C                       4 times the image mean intensity.
C
C     CLIP            [default] Prevent roll-over of the intensity mapping.
C     NOCLIP          [optional] enable LUT rollover.  CLIP is retained for
C                       backwards compatibility with old scripts.
C
C     POSITIVE        [optional] Make the hardcopy White-on-Black background.
C                       The default mapping is Black-on-White.
C
C     TITLE           [optional] Put the Image title (in FITS header OBJECT
C                       card) on the hardcopy.
C
C     HIST            [optional] Map the image by performing a histogram 
C     HIST=xxx          equalization.  Options: (FLAT,SQRT,LOG) [Default: FLAT]
C
C     LAND            [optional] Output will appear in LANDSCAPE mode, paper
C                       oriented long-side horizontal.  Default is with paper
C                       oriented long-side vertical (portrait).
C
C     FOUR            [optional] Use a 4 bit (16 level) grayscale rather
C                       than the default 8 bit (256 level) grayscale.
C
C     BLACK=nnn       [optional] Set the value for black saturation for
C                       the printing device to nnn, where nnn<=255.  
C                       Default is 255 for 8-bit plots.  Ignored if 4-bit plot
C
C     AXES            [optional] Puts coordinates axes on the image.
C
C     NOBOX           [optional] Supresses all axis plotting (including box)
C
C     COMMENT         [optional] Comment to append to image hardcopy, may
C     COMMENT=          be up to 64 characters long.  May be specified on
C			the command line (COMMENT=) or via a user prompt
C			[default].  If the on-line comment contains more than
C			1 word, the string must be contained in single quotes.
C
C     FILE=xxx        [optional] Direct the PostScript Image into a file 
C     OUT=xxx           named xxx.  By default, IMPOST writes the image into 
C                       IMAGE.PS in the user`s current working directory.
C
C     SCALE=s         [optional] specify the pixel scale in units/pixel
C                       for the image.  The axis will be labelled in these
C                       units rather than pixels (default, see AXES keyword).
C                       The origin is assumed to be the image center unless
C                       the CEN=(r,c) keyword is used.
C
C     CEN=(r,c)       [optional] specify the center of the image in
C                       pixels for use with the SCALE=s keyword.  Default 
C                       center is the image physical center.
C
C     FLIP            [optional] flips output in rows so origin is in 
C			the lower left corner (reverse raster order, or
C			the order assumed by IRAF and STSDAS).
C
C     NOBAR           [optional] No intensity bar is to be drawn
C     
C     BAR=XXX         [optional] User defines the intensity units
C                       to label the intensity bar.  Default is
C                       "Intensity".
C
C     INFO	      [optional] Write in a line of auxilliary info along
C                       the bottom of the page in small type.  Not all
C                       keywords generate info line entries.
C
C     EPS             [optional] Write an Encapsulated PostScript file
C                       for embedding in a document.  EPS files do not print
C                       "standalone" but only after embedding in a file.
C
C  -- Advanced Page Control --
C
C     WIND=(w,h)      [optional] specify the maximum possible size of the
C                       plotting window, in inches.  The window is the largest
C                       region in which IMPOST will try to fit the image.
C                       By default, the plotting window is:
C                          Portrait:  6" x 6"
C                          Landscape: 8" x 5"
C                       Dimensions are specified as "width","height" in units
C                       of inches.  Must be smaller than the paper dimensions 
C                       (see PAGE keyword).
C
C     ORIGIN=(x,y)    [optional] specify the origin of the plotting window
C                       in inches from the lower left-hand corner of the 
C                       paper.  X is horizontal, Y is vertical.  By default,
C                       IMPOST tries to center the plotting window on the page,
C                       and adds a 30pt binding margin to Portrait mode 
C                       plots.  ORIGIN overrides this binding margin.
C
C     PAGE=(L,S)      [optional] specify the physical size of the paper in
C                       inches.  "L" = long dimension, "S" = short dimension.
C                       For example, PAGE=(11,8.5) is for standard letter
C                       paper.  Used only if you have non-standard sized paper
C                       in your local PostScript printer.
C
C     COPIES=n     	specifies number of copies to output
C
C     LARGE           Alternative to WIND=, will make the plot window
C			as large as possible for the given page size.
C
C  -----------------------------------------------
C
C    Author:   Richard Pogge          1987 June 14
C              Lick Observatory
C              University of California
C              Santa Cruz, CA  95064
C
C    Version 1.0:  1987 June 14
C            1.1:  1988 July 17 - Sun Fortran f77 compatibility
C            1.2:  1988 October 20  VISTA Version 3.5 Upgrade
C            1.3:  1990 April 22 - Added histogram equalization 
C            1.4:  1990 June 7 - Added FILE, SCALE, and CEN keywords.
C    Version 2.0:  1990 June 9 - Interfaced with LickMongo.
C            2.1:  1990 July 17 - Added PAGE, WIND, and ORIGIN keywords
C            2.2:  1990 December 4 - Added FLIP, LARGE, and COPIES=
C                   keywords ala` Jon Holtzman and incorporated all his 
C                   other little hooks and goodies.  = VISTA v4.1 upgrade.
C            2.3:  1991 January 20 - added BAR= and NOBAR keywords.  Fixed
C                   bugs in ORIGIN, FLIP, FILE and OUT keywords.
C            2.4:  1992 August 17 - added BLACK= keyword to allow compensation
C                   for black color saturation on a particular PostScript
C                   device (poor-astronomer`s gamma correction), and modified
C                   the color bar plotting appropriately for 8-bit plots. [RWP]
C            2.5:  1995 Mar 26 - introduced EPS keyword to emit the file
C                   as Encapsulated PostScript.  Uses the EPS utilities in
C                   the latest LickMongo release [rwp]
C            2.6:  1995 July 3 - added NOBOX keyword from Lowell version
C                   2.3 [rwp]
C            2.7:  1996 April 23 - Fixed raster error when binning odd-sized
C                   images, and added NOCLIP keyword, making CLIP default
C                   (better mirrors behavior of the X11 TV command) [rwp]
C            2.8:  1997 Aug 11 - added MAXBIN parameter and set to 1024
C                   new laser printers can handle larger images without
C                   having to bin, so parameterized the binning threshold 
C                   [rwp]
C
C =======================================================
C
C    Version 3.0:  1998 Feb 27 - Name was changed from POSTIT to IMPOST
C                   because of a letter of 1997 Nov 21 to Patrick Osmer
C                   (Dept. Chair at OSU Astronomy) from the [butthead] 
C                   lawyers at the law firm of Arnold, White, & Durkee.
C                   This firm represents the Minnesota Mining & Manufacturing
C                   Company (3M) in trademark and other intellectual 
C                   property rights matters.  It seems after coming across
C                   a web version of the Prospero manual (a VISTA spin-off
C                   written at OSU that incorporates POSTIT because yours
C                   truly wrote it), they deemed our use of the word POSTIT
C                   was infringing the trademark of 3M for its famous
C                   yellow sticky notes, aka Post-It(tm) Notes.
C
C                   Since 3M is a whole lot bigger and badder than the Ohio
C                   State University, our butthead lawyers decided that we
C                   should cave in to their butthead lawyers, and so 
C                   we had to change the name.
C
C                   Alternative names suggested by a non-statistical poll
C                   of VISTA users included:
C                      SCOTCHTAPE
C                      3MSUCKS (variant: 3MSUX)
C                      SLIMEBALL_LAWYERS
C                      BHL (Butt-Head Lawyers - as if you had to ask...)
C                      THE_COMMAND_FORMERLY_KNOWN_AS_POSTIT (the DT memorial
C                          alternative command name)
C                      XEROX
C                      KLEENEX
C                      FRIDGE
C                      COKE
C                    and the list got worse from there.

C
C ----------------
C
C  Future Developments:
C
C      The following future revisions are being contemplated:
C
C          1) HREJECT option, to allow the user to reject the top and bottom
C             fractions of the intensity points when making histogram scaling
C             of the image.  Syntax would be HREJECT=(bot,top), and the
C             arguments would be in units of the fraction of the total number
C             of points.  For example, HREJECT=(0.0,0.1) means no low-end
C             rejection and rejection of 10% of the high-end.
C
C          2a) Compiler flags PSINCHES and PSCMS to set the internal units to 
C             be either Inches or Centimeters to better accomodate both
C             installations in countries which use either English or Metric
C             units.  The laser printers will continue to use printers points,
C             despite the native units of where they happen to be installed.
C
C          2b) Compilier flags LETTER and A4PAGE to set the default page
C             size to be either Letter (8.5x11-inches) or A4 (21x29.7cm)
C             paper sizes.
C
C-----------------------------------------------------------------------------
 
      REAL  A(ISCOL:IECOL,ISROW:IEROW)
      
#ifdef VMS
      include 'VINCLUDE:vistalink.inc'
      include 'VINCLUDE:imagelink.inc'
      include 'VINCLUDE:customize.inc'
#else   /* VMS */
      INCLUDE 'vistadisk/source/include/vistalink.inc'
      INCLUDE 'vistadisk/source/include/imagelink.inc'
      INCLUDE 'vistadisk/source/include/customize.inc'
#endif  /* VMS */
 
      INTEGER ROW, COL, NROW, NCOL
      INTEGER BOXID
      INTEGER SR, ER, SC, EC
      INTEGER NAVG, NRAVG, NCAVG
 
C   array into which each group of 40 (or 80) pixels is packed
 
      INTEGER IPIX(80)
 
C   Image scale and center
 
      REAL PSCALE
      REAL CCEN, RCEN
      REAL RPIX(2)
      INTEGER NPIX
      REAL RMIN, RMAX
      REAL CMIN, CMAX
      
C   average intensity of a given pixel cluster
 
      REAL APIX
 
C   number of gray levels
 
      INTEGER NLEV
 
C   integral compression (binning) factor
 
      INTEGER IBIN
 
C   dimensions of paper (in printers points and inches, resp.)
 
      REAL WPAGE, HPAGE
      REAL PLONG, PSHORT
 
C   dimension of image (in points)
 
      REAL WIDTH, HEIGHT
 
C   largest image size (in points and inches, resp.)
 
      REAL WMAX, HMAX
      REAL WWID, WHEI
 
C   Image Left and Bottom Margins:
 
      REAL WMARGIN, HMARGIN
 
C   location of bounding box (llx,lly & urx,ury) in points
 
      REAL W0, H0
      REAL WM, HM

C   Location of bounding box for intensity bar in points, and scaling

      REAL XBAR, XBARM
      REAL YBAR, YBARM
      REAL BLMIN, BLMAX

C   Image title, comment, and intensity bar label and lengths in chars
 
      CHARACTER IMTITLE*64
      INTEGER LTITLE
 
      CHARACTER*80 IMCOM
      INTEGER LCOM

      CHARACTER*80 BARLAB
      INTEGER LBAR

C   Information Line string
      
      CHARACTER*120 INFOLINE, FSTRCAT
      INTEGER LINFO

#ifdef __SUN3
C   Hex stuff for Sun 3
      CHARACTER OUTLINE*80, HH*2
#endif

C   User defined file name
 
      CHARACTER*80 EPSNAME, EPSFILE
      CHARACTER*4  EPSEXT
      LOGICAL DOEPS
      INTEGER LPSF
 
C   Histogram equalization declarations
 
      INTEGER*4  HEMIN, HEMAX
      PARAMETER  (HEMIN = -32768)
      PARAMETER  (HEMAX = 37267)
 
      INTEGER*4  IMHIST(HEMIN:HEMAX)
      INTEGER*4  IMLUT(HEMIN:HEMAX)

      COMMON /WORK/ IMHIST, IMLUT

      INTEGER*4  IHST
      INTEGER*4  DSPMIN, DSPMAX
 
      REAL*4     IMMAX, IMMIN
      REAL*4     BZERO, BSCALE
      REAL*4     HISTMIN, HISTMAX
      REAL*4     SUM
      LOGICAL    HISTEQ
      INTEGER*4  HISTOP
 
C   Various working variables
 
      REAL*4    G(2)
      INTEGER   INNPAR

      INTEGER   IBAR(33)
      CHARACTER FMTSTR*8
 
      CHARACTER*80 PARM, TWORD
      CHARACTER*40 MONGOCOM(4), MACROFILE
      CHARACTER*500 TCOM
      CHARACTER*80 PSCMD
 
      LOGICAL CLIP, POSITIVE, TITLE, AXES, LARGE, FLIP, INTERACTIVE
      LOGICAL USERSPAN, LANDSCAPE, FOUR, COMMENT, MACRO
      LOGICAL IMSCALE, IMCEN, NOFILE, BAR, INFO
      LOGICAL UORIGIN, BINOK, USEBOX

C  External Functions

      INTEGER PMGO(FILEPLOT), UPPER
      LOGICAL PTUV(PSOPEN)

c      LOGICAL KEYCHECK
 
C   Check Keywords
 
      CALL KEYINIT
      CALL KEYDEF('BOX=')
      CALL KEYDEF('Z=')
      CALL KEYDEF('L=')
      CALL KEYDEF('CLIP')
      CALL KEYDEF('NOCLIP')
      CALL KEYDEF('POSITIVE')
      CALL KEYDEF('TITLE')
      CALL KEYDEF('HIST')
      CALL KEYDEF('HIST=')
      CALL KEYDEF('LAND')
      CALL KEYDEF('FOUR')
      CALL KEYDEF('BLACK=')
      CALL KEYDEF('AXES')
      CALL KEYDEF('COMMENT')
      CALL KEYDEF('SCALE=')
      CALL KEYDEF('FILE=')
      CALL KEYDEF('CEN=')
      CALL KEYDEF('WIND=')
      CALL KEYDEF('ORIGIN=')
      CALL KEYDEF('PAGE=')
      CALL KEYDEF('LARGE')
      CALL KEYDEF('FLIP')
      CALL KEYDEF('COMMENT=')
      CALL KEYDEF('COPIES=')
      CALL KEYDEF('OUT=')
      CALL KEYDEF('NOBAR')
      CALL KEYDEF('BAR=')
      CALL KEYDEF('INFO')
      CALL KEYDEF('INT')
      CALL KEYDEF('MACRO=')
      CALL KEYDEF('NOBIN')
      CALL KEYDEF('EPS')
      CALL KEYDEF('MAXSIZE=')

C   If the image display is to be within a given box, get the relevant info.
C   Also see if we are using LANDSCAPE or LANDSCAPE mode, as this will
C   affect certain scaling defaults
 
      BOXID = 0
      LANDSCAPE = .FALSE.
      INFO = .FALSE.
      USEBOX = .TRUE.
      DO 8701 I=1, NWORD
	 TWORD = WORD(I)
	 L = UPPER(TWORD)
 
C   User-defined BOX to be used?
 
         IF (TWORD(1:4) .EQ. 'BOX=') THEN
            CALL ASSIGN(TWORD,F,PARM)
            BOXID = NINT(F)
            IF (XERR) RETURN
 
C   Orient the paper long-side horizontal (LANDSCAPE mode)
 
         ELSE IF (TWORD(1:4) .EQ. 'LAND') THEN
            LANDSCAPE = .TRUE.
            
C   Don`t draw a box around the image
 
         ELSE IF (TWORD(1:5) .EQ. 'NOBOX') THEN
            USEBOX = .FALSE.
            
C   Add a line of auxillary information to the bottom of the page

         ELSE IF (TWORD(1:4) .EQ. 'INFO') THEN
            INFO = .TRUE.
            INFOLINE = 'VISTA IMPOST 3.0:'
         END IF
 8701 CONTINUE
 
      IF (BOXID .EQ. 0) THEN
         SR = ISROW
         SC = ISCOL
         ER = IEROW
         EC = IECOL
      ELSE
         CALL GETBOX ( BOXID, ICOORD(1,IM), SR, ER, SC, EC )
         IF (XERR) RETURN
      END IF
 
C   Compute the size of the image to be displayed in rows and columns
 
      NROW = ER - SR + 1
      NCOL = EC - SC + 1
 
C   Establish default values
 
C   Logical Assignments (* = default):
C
C      Logical            .TRUE.                .FALSE.
C
C     CLIP         * disable LUT rollover     enable LUT rollover
C     POSITIVE       White on Black         * Black on White
C     LANDSCAPE      Landscape Mode         * Portrait Mode
C     TITLE          Title Image            * No Image Title
C     USERSPAN       User intensity span    * compute intensity span
C     HISTEQ         Histogram equalization * Normal linear mapping
C     FOUR           4-bit (16 level)       * 8-bit (256 level)
C                    gray scale               gray scale
C     AXES           Draw axis ticks        * No axis ticks
C     BAR          * Draw intensity wedge     No intensity wedge
C     COMMENT        Add comment text       * No comment text
C     NOFILE       * Default file name        User file name
C     IMSCALE        User image scale       * No scale, axes=pixels
C     IMCEN          User origin            * Image Center=Origin
C     UORIGIN        User plot window       * Default plot window origin
C                      origin       
C     FLIP           Raster botom-to-top    * Raster Top-to-bottom
C     LARGE          Fill paper with image  * Normal default window
C     INFO           Write auxillary info   * No information line
C     INTERACTIVE    Interactive Mongo Mode * no interactive Mongo mode
C     MACRO          Plot a Mongo macro     * no Mongo macro to plot
C     BINOK        * Bin if >MAXSIZE        suppress binning
C     DOEPS          Encapsulated PS        * printable PostScript
C     

      CLIP = .TRUE.
      POSITIVE = .FALSE.
      TITLE = .FALSE.
      USERSPAN = .FALSE.
      HISTEQ = .FALSE.
      FOUR = .FALSE.
      AXES = .FALSE.
      BAR = .TRUE.
      COMMENT = .FALSE.
      NOFILE = .TRUE.
      IMSCALE = .FALSE.
      IMCEN = .FALSE.
      UORIGIN = .FALSE.
      FLIP = .FALSE.
      LARGE = .FALSE.
      INTERACTIVE = .FALSE.
      MACRO = .FALSE.
      BINOK = .TRUE.
      MAXSIZE = 1024
      DOEPS = .FALSE.
      EPSEXT = '.ps'

      NLEV = 256
      ZERO = 0.0
      CCEN = FLOAT(SC) + FLOAT(NCOL)/2.0
      RCEN = FLOAT(SR) + FLOAT(NROW)/2.0

      BARLAB = 'Intensity'
      LBAR = 9
      MACROFILE = ' '

C   Default paper (page) and plotting window sizes
      PLONG = 11.0
      PSHORT = 8.5
 
      IF (LANDSCAPE) THEN
         WWID = 8.0
         WHEI = 5.5
      ELSE
         WWID = 6.0
         WHEI = 6.0
      END IF

      NCOPIES = 1
 
C   Parse the keywords given, and set logical as appropriate
 
      DO 8702 I=1, NCON

	 TWORD = WORD(I)
	 L = UPPER(TWORD)
 
C     User has specified the intensity map zero point.
 
         IF (TWORD(1:2) .EQ. 'Z=') THEN
            CALL ASSIGN(TWORD,F,PARM)
            IF (XERR) RETURN
            ZERO = F
            IF (INFO) THEN
               LINFO = NUMCHAR(INFOLINE)
               INFOLINE = FSTRCAT(INFOLINE(1:LINFO),FSTRCAT(' ',TWORD))
               LINFO = NUMCHAR(INFOLINE)
            END IF
            
C     User has specified the intensity map span.
 
         ELSE IF (TWORD(1:2) .EQ. 'L=') THEN
            CALL ASSIGN(TWORD,F,PARM)
            SPAN = F
            USERSPAN = .TRUE.
            IF (INFO) THEN
               LINFO = NUMCHAR(INFOLINE)
               INFOLINE = FSTRCAT(INFOLINE(1:LINFO),FSTRCAT(' ',TWORD))
               LINFO = NUMCHAR(INFOLINE)
            END IF
 
C     User has requested intensity map rollover
 
         ELSE IF (TWORD(1:6) .EQ. 'NOCLIP') THEN
            CLIP = .FALSE.
            IF (INFO) THEN
               LINFO = NUMCHAR(INFOLINE)
               INFOLINE = FSTRCAT(INFOLINE(1:LINFO),' NOCLIP')
               LINFO = NUMCHAR(INFOLINE)
            END IF
 
C     User wants the image to be printed White-on-Black background
 
         ELSE IF (TWORD(1:8) .EQ. 'POSITIVE') THEN
            POSITIVE = .TRUE.
            IF (INFO) THEN
               LINFO = NUMCHAR(INFOLINE)
               INFOLINE = FSTRCAT(INFOLINE(1:LINFO),' POSITIVE')
               LINFO = NUMCHAR(INFOLINE)
            END IF

C     User wants the title plotted.
 
         ELSE IF (TWORD(1:5) .EQ. 'TITLE') THEN
            TITLE = .TRUE.
            CALL CCCHEAD('OBJECT',HEADBUF(1,IM),IMTITLE)
            LTITLE = NUMCHAR(IMTITLE)
            IF (LTITLE .GT. 64) LTITLE = 64
 
C     Do a histogram equalization for display
 
         ELSE IF (TWORD(1:5) .EQ. 'HIST=') THEN
            HISTEQ = .TRUE.
            IF (TWORD(6:8) .EQ. 'LOG') THEN
               HISTOP = 2
            ELSE IF (TWORD(6:9) .EQ. 'SQRT') THEN
               HISTOP = 1
            ELSE
               HISTOP = 0
            END IF
            IF (INFO) THEN
               LINFO = NUMCHAR(INFOLINE)
               INFOLINE = FSTRCAT(INFOLINE(1:LINFO),FSTRCAT(' ',TWORD))
               LINFO = NUMCHAR(INFOLINE)
            END IF
            
         ELSE IF (TWORD(1:4) .EQ. 'HIST') THEN
            HISTEQ = .TRUE.
            HISTOP = 0
            IF (INFO) THEN
               LINFO = NUMCHAR(INFOLINE)
               INFOLINE = FSTRCAT(INFOLINE(1:LINFO),' HIST=LIN')
               LINFO = NUMCHAR(INFOLINE)
            END IF
 
C     User wants to use a 4-bit (16 level) grayscale representation.
 
         ELSE IF (TWORD(1:4) .EQ. 'FOUR') THEN
            FOUR = .TRUE.
            NLEV = 16
            IF (INFO) THEN
               LINFO = NUMCHAR(INFOLINE)
               INFOLINE = FSTRCAT(INFOLINE(1:LINFO),' 4-Bit')
               LINFO = NUMCHAR(INFOLINE)
            END IF

C     User needs to set a black color saturation level smaller than 255
C     for 8-bit plots

         ELSE IF (TWORD(1:6) .EQ. 'BLACK=') THEN
            CALL ASSIGN(TWORD,F,PARM)
            IF (XERR) RETURN
            NLEV = MIN(256,NINT(F))
            
C     User wants to draw axes
 
         ELSE IF (TWORD(1:4) .EQ. 'AXES') THEN
            AXES = .TRUE.
 
C     User wants to suppress autobinning for images >MAXSIZE pixels
 
         ELSE IF (TWORD(1:5) .EQ. 'NOBIN') THEN
            BINOK = .FALSE.

         ELSE IF (TWORD(1:8) .EQ. 'MAXSIZE=') THEN
            CALL ASSIGN(TWORD,F,PARM)
            IF (XERR) RETURN
            MAXSIZE = NINT(F)
 
C     User wants to add a comment line.
 
         ELSE IF (TWORD .EQ. 'COMMENT') THEN
            COMMENT = .TRUE.
            PRINT '(1X,''Comment ? : '' $)'
            READ (*, '(A)') IMCOM
            LCOM = NUMCHAR(IMCOM)
            IF (LCOM .GT. 80) LCOM = 80

	 ELSE IF (TWORD(1:8) .EQ. 'COMMENT=') THEN
	    COMMENT = .TRUE.
	    TCOM = COMMAND
	    L = UPPER(TCOM)
	    L = INDEX(TCOM,'COMMENT=')
	    CALL DISSECT(COMMAND(L:),2,.TRUE.,NTYPE,NUM,FNUM,
     &                   IMCOM, LCOM, XERR)
	    IF (XERR) RETURN
	    IF (LCOM .GT. 80) LCOM = 80
 
C     User-defined file name
 
         ELSE IF (TWORD(1:5) .EQ. 'FILE=') THEN
            NOFILE = .FALSE.
            LPSF = NUMCHAR(TWORD)
            IF (LPSF .EQ. 5) THEN
               PRINT *, '**ERROR: No file name given'
               XERR = .TRUE.
               RETURN
            END IF
            EPSFILE = WORD(I)(6:LPSF)
 
         ELSE IF (TWORD(1:4) .EQ. 'OUT=') THEN
            NOFILE = .FALSE.
            LPSF = NUMCHAR(TWORD)
            IF (LPSF .EQ. 4) THEN
               PRINT *, '**ERROR: No file name given'
               XERR = .TRUE.
               RETURN
            END IF
            EPSFILE = WORD(I)(5:LPSF)
 
C     Scale the image by the user-defined pixel scale
 
         ELSE IF (TWORD(1:6) .EQ. 'SCALE=') THEN
            CALL ASSIGN(TWORD,F,PARM)
            PSCALE = F
            IF (PSCALE .EQ. 0.0) THEN
               PRINT *,'**ERROR: SCALE must be nonzero'
               XERR = .TRUE.
               RETURN
            END IF
            IMSCALE = .TRUE.
            AXES = .TRUE.
            IF (INFO) THEN
               LINFO = NUMCHAR(INFOLINE)
               INFOLINE = FSTRCAT(INFOLINE(1:LINFO),FSTRCAT(' ',TWORD))
               LINFO = NUMCHAR(INFOLINE)
            END IF
 
C     User-defined image center (for SCALE)
 
         ELSE IF (TWORD(1:4) .EQ. 'CEN=') THEN
            CALL ASSIGNV(TWORD,2,RPIX,NPIX,PARM)
            IF (XERR) RETURN
            IF (NPIX .LT. 2) THEN
               PRINT *,'**ERROR: One center coordinate not given...'
               XERR = .TRUE.
               RETURN
            END IF
            RCEN = RPIX(1)
            CCEN = RPIX(2)
            IMCEN = .TRUE.
            IF (INFO) THEN
               LINFO = NUMCHAR(INFOLINE)
               INFOLINE = FSTRCAT(INFOLINE(1:LINFO),FSTRCAT(' ',TWORD))
               LINFO = NUMCHAR(INFOLINE)
            END IF
 
C     User wishes to change the plotting window size
 
         ELSE IF (TWORD(1:5) .EQ. 'WIND=') THEN
            CALL ASSIGNV(TWORD,2,G,INNPAR,PARM)
            IF (XERR) RETURN
            IF (INNPAR .LT. 2) THEN
               PRINT *,'**ERROR: One window dimension not given...'
               XERR = .TRUE.
               RETURN
            END IF
            WWID = G(1)
            WHEI = G(2)

C     User wants the maximum window size

	 ELSE IF (TWORD .EQ. 'LARGE') THEN
	     LARGE = .TRUE.
 
C     User wishes to change the plotting origin.  Assumes the user`s
C     coordinates are in units of INCHES, and converts them to
C     PostScript points (72 pts/inch).
 
         ELSE IF (TWORD(1:7) .EQ. 'ORIGIN=') THEN
            CALL ASSIGNV(TWORD,2,G,INNPAR,PARM)
            IF (XERR) RETURN
            IF (INNPAR .LT. 2) THEN
               PRINT *,'**ERROR: One origin coordinate not given...'
               XERR = .TRUE.
               RETURN
            END IF
            W0 = G(1)*72.0
            H0 = G(2)*72.0
            IF (W0 .LT. 0.0 .OR. H0 .LT. 0.0) THEN
               PRINT *, '** Origin coordinates must be positive'
               XERR = .TRUE.
               RETURN
            END IF
            UORIGIN = .TRUE.
 
C     User wishes to change the paper size (assumes user`s dimensions
C     are INCHES).
 
         ELSE IF (TWORD(1:5) .EQ. 'PAGE=') THEN
            CALL ASSIGNV(TWORD,2,G,INNPAR,PARM)
            IF (XERR) RETURN
            IF (INNPAR .LT. 2) THEN
               PRINT *,'**ERROR: One page dimension not given...'
               XERR = .TRUE.
               RETURN
            END IF
            PLONG = G(1)
            PSHORT = G(2)
            IF (PLONG .LT. PSHORT) THEN
               TEMP = PLONG
               PLONG = PSHORT
               PSHORT = TEMP
            END IF

C     User wants to set the number of copies

	 ELSE IF (TWORD(1:7) .EQ. 'COPIES=') THEN
	    CALL ASSIGN(TWORD,TEMP,PARM)
	    IF (XERR) RETURN
	    NCOPIES = NINT(TEMP)

C     User wants to flip the raster order

	 ELSE IF (TWORD .EQ. 'FLIP') THEN
	    FLIP = .TRUE.
            IF (INFO) THEN
               LINFO = NUMCHAR(INFOLINE)
               INFOLINE = FSTRCAT(INFOLINE(1:LINFO),' FLIP')
               LINFO = NUMCHAR(INFOLINE)
            END IF
 
C     User want no intensity wedge drawn

         ELSE IF (TWORD .EQ. 'NOBAR') THEN
            BAR = .FALSE.

C     User wants to change the default intensity bar label 

         ELSE IF (TWORD(1:4) .EQ. 'BAR=') THEN
            BAR = .TRUE.
	    TCOM = COMMAND
	    L = UPPER(TCOM)
	    L = INDEX(TCOM,'BAR=') 
	    CALL DISSECT(COMMAND(L:),2,.TRUE.,NTYPE,NUM,FNUM,
     &                   BARLAB, LBAR, XERR)
	    IF (XERR) RETURN
	    IF (LBAR .GT. 80) LBAR = 80

C     Allow user to enter interactive mongo mode (backwards compatibility
C     mode only, use of INT is deprecated because you can`t see)

	 ELSE IF (TWORD .EQ. 'INT') THEN
	    INTERACTIVE = .TRUE.

C     After plotting, execute the macrofile given on the command line

	 ELSE IF (TWORD(1:6) .EQ. 'MACRO=') THEN
	    MACRO = .TRUE.
	    MACROFILE = WORD(I)(7:)

C     Create Encapsulated PostScript instead of printable PostScript

	 ELSE IF (TWORD(1:3) .EQ. 'EPS') THEN
	    DOEPS = .TRUE.
            EPSEXT = '.eps'

         END IF
 
 8702 CONTINUE
 
C   If there are unknown keywords, exit gracefully.
 
c     IF (.NOT. KEYCHECK()) THEN
c         XERR = .TRUE.
c         RETURN
c      END IF

C   Check incompatible keywords

      IF (FOUR .AND. NLEV .NE. 16) NLEV=16
 
C   If a LARGE page is to be used, set the window according to the 
C   paper size.  LARGE pages have 0.5'' margins (at least) around the
C   image.

      IF (LARGE) THEN
         IF (LANDSCAPE) THEN
            WWID = PLONG - 1.0
            WHEI = PSHORT - 1.0
         ELSE
            WWID = PSHORT - 1.0
            WHEI = PLONG - 1.0
         END IF
      END IF

C   Check to make sure page dimensions are sensible

      IF (LANDSCAPE) THEN
         IF (WWID .GT. PLONG) THEN
            PRINT *, '**ERROR: Window wider than paper'
            XERR = .TRUE.
            RETURN
         ELSE IF (WHEI .GT. PSHORT) THEN
            PRINT *, '**ERROR: Window higher than paper'
            XERR = .TRUE.
            RETURN
         END IF
      ELSE
         IF (WWID .GT. PSHORT) THEN
            PRINT *, '**ERROR: Window wider than paper'
            XERR = .TRUE.
            RETURN
         ELSE IF (WHEI .GT. PLONG) THEN
            PRINT *, '**ERROR: Window higher than paper'
            XERR = .TRUE.
            RETURN
         END IF
      END IF
 
C   If one (or both) of the image dimensions is greater than MAXSIZE pixels, bin
C   (compress) the image before display.  Large images require a prohibitive
C   amount of time to display, and may overrun the PostScript printer`s memory.
C   In addition, the printer`s resolution may be insufficient to the task,
C   so the extra data would be wasted.  We can suppress this with the 
C   NOBIN keyword if the user has a fancy printer with tons of memory or is
C   just feeling lucky that day.
 
      IF (BINOK .AND. (NROW .GT. MAXSIZE .OR. NCOL .GT. MAXSIZE)) THEN
         IBINC = (NCOL - 1) / MAXSIZE + 1
         IBINR = (NROW - 1) / MAXSIZE + 1
         IBIN   = MAX ( IBINR, IBINC )
         PRINT *, 'WARNING:  Image being COMPRESSED by ',IBIN
      ELSE
         IBIN = 1
      END IF
 
C   Build the PostScript file name.

      IF (NOFILE) THEN
         CALL FILEDEF('image',EPSNAME,' ',EPSEXT)
      ELSE
         CALL FILEDEF(EPSFILE,EPSNAME,' ',EPSEXT)
      END IF
      IF (XERR) THEN
         PRINT *,'**ERROR: Filename Syntax Error'
         RETURN
      END IF
 
C   If the "SPAN=" keyword is not found, compute the mean of the image, and 
C   use 4x that for the intensity map span.  Use every SQRT(N)-th pixel for 
C   speed.
 
      IF (.NOT. HISTEQ) THEN
         IF (.NOT. USERSPAN) THEN
            NRAVG = NROW
            NCAVG = NCOL
            IF (NROW .GT. 50) NRAVG = SQRT(FLOAT(NROW))
            IF (NCOL .GT. 50) NCAVG = SQRT(FLOAT(NCOL))
            SPAN = 0.0
            NAVG = 0
            DO 8703 ROW = SR, ER, NRAVG
               DO 8704 COL = SC, EC, NCAVG
                  NAVG = NAVG + 1
                  SPAN = A(COL,ROW) + SPAN
 8704          CONTINUE
 8703       CONTINUE
            SPAN = SPAN / NAVG
         END IF
 
      ELSE 
      
C   Compute Histogram Equalization LUT
 
C   Find the image limits and compute scaling factors to map A into
C   16 bits.
 
         IF (NOGO) RETURN
         PRINT *, 'Computing Image Limits...'
         IMMIN = A(SC,SR)
         IMMAX = IMMIN
         DO ROW = SR, ER
            DO COL = SC, EC                    
               APIX = A(COL,ROW)
	       IMMIN = MIN(APIX,IMMIN)
               IMMAX = MAX(APIX,IMMAX)
            END DO
         END DO
 
         HISTMIN = FLOAT(HEMIN)
         HISTMAX = FLOAT(HEMAX)
         BZERO = (IMMAX * HISTMIN - IMMIN * HISTMAX)
     &                         / (HISTMAX - HISTMIN)
         IF (IMMAX .EQ. IMMIN) THEN
            BSCALE = 1
         ELSE
            BSCALE = (HISTMAX-HISTMIN)/(IMMAX-IMMIN)
         END IF
 
C   Load the image histogram
 
         PRINT *, 'Loading Image Histogram...'
         DO I = HEMIN, HEMAX
            IMHIST(I) = 0
         END DO
         IF (NOGO) RETURN
         DO ROW = SR, ER
            DO COL = SC, EC
               APIX = (A(COL,ROW)+BZERO)/BSCALE
               IHST = NINT(APIX)
               IHST = MAX(MIN(IHST,HEMAX),HEMIN)
               IMHIST(IHST) = 1 + IMHIST(IHST)
            END DO
         END DO
 
C   Generate the histogram lookup table
 
         IF (NOGO) RETURN
         PRINT *, 'Loading the LUT...'
 
C      Compute the intensity limits of interest, DSPMIN & DSPMAX
 
         DSPMIN = NINT((ZERO+BZERO)/BSCALE)
         DSPMIN = MAX(MIN(DSPMIN,HEMAX),HEMIN)
 
         IF (USERSPAN) THEN
            DSPMAX = NINT((ZERO+SPAN+BZERO)/BSCALE)
            DSPMAX = MAX(MIN(DSPMAX,HEMAX),HEMIN)
         ELSE
            DSPMAX = HEMAX
         END IF
 
         IF (DSPMAX .LT. HEMAX) THEN
            DO I = HEMAX, DSPMAX+1, -1
               IMLUT(I) = NLEV - 1
            END DO
         END IF
             
C      Compute the number of pixels with intensities between the intensity
C      limits of interest
 
         IF (NOGO) RETURN
         SUM = 0.0
         DO I = DSPMIN, DSPMAX
            SUM = SUM + IMHIST(I)
         END DO
 
         IF (HISTOP .EQ. 2) THEN
            SUM = ALOG (SUM)
         ELSE IF (HISTOP .EQ. 1) THEN
            SUM = SQRT (SUM)
         END IF
         HFAC = FLOAT(NLEV-1) / SUM
 
C      Load the LUT such that for each display intensity in the range 
C      [0,NLEV-1] there are an equal number of pixels with that
C      intensity  (that's why it's called a "histogram equalization"...)
 
         IF (NOGO) RETURN
         SUM = 1.0
         DO I = DSPMAX, HEMIN, -1        
            SUM = SUM + FLOAT(IMHIST(I))
            IF (HISTOP .EQ. 1) THEN
               APIX = SQRT (SUM)
            ELSE IF (HISTOP .EQ. 2) THEN
               APIX = ALOG (SUM)
            ELSE 
               APIX = SUM
            END IF
            IMLUT(I) = (NLEV-1) - NINT(HFAC * APIX)
            IF (IMLUT(I) .GT. (NLEV-1)) IMLUT(I) = NLEV - 1
            IF (IMLUT(I) .LT. 0) IMLUT(I) = 0
         END DO
 
      END IF
 
C   We will work through LickMongo`s PostScript drivers.
C   Open a LickMongo PostScript file.

      IF (DOEPS) THEN
         CALL PMGO(EPSPLOT)(LANDSCAPE,EPSNAME,' ')
      ELSE
         CALL PMGO(PSPLOT)(LANDSCAPE,EPSNAME,' ')
      END IF

C   Set the PostScript mode to only use the internal PS font tables
C   stored in the printer`s ROM.  This saves a great deal of time
C   and vectors.

      CALL PMGO(SETPSFMODE)(2)

C   Now, issue a PTUV(PSOPEN) to open the PostScript file and set the logical
C   unit for it.  We do this because we will issue PTUV(pswrite) calls before
C   any other LickMongo calls.  Otherwise, all subsequent PTUV(pswrite) calls
C   would get sent to STDOUT.

      IF (.not. PTUV(psopen)()) then
         L = NUMCHAR(EPSNAME)
         PRINT *,'Cannot open output PostScript file ',EPSNAME(1:L)
         PRINT *,'Possibilities:'
         PRINT *,' 1) You have no write permission in this directory'
         PRINT *,' 2) You gave an invalid filename/directory path'
         PRINT *,' 3) You are out of disk space'
         PRINT *,' '
         xerr = .true.
         return
      end if

C   The next lines are comments about how the image was processed prior
C   to being written into the PostScript file.
 
      CALL PTUV( pswrite)('%Image Parameters:')
      IF (HISTEQ) THEN
         IF (HISTOP .EQ. 2) THEN
            PSCMD = '%Mapping: Logarithmic Histogram'
         ELSE IF (HISTOP .EQ. 1) THEN
            PSCMD = '%Mapping: Square-Root Histogram'
         ELSE
            PSCMD = '%Mapping: Histogram Equalization'
         END IF
         CALL PTUV( pswrite)(PSCMD)
      ELSE
         CALL PTUV( pswrite)('%Mapping: Linear')
         IF (CLIP) THEN
            CALL PTUV( pswrite)('%Rollover: Clipped [default]')
         ELSE
            CALL PTUV( pswrite)('%Rollover: Enabled')
         END IF
      END IF
 
      IF (POSITIVE) THEN
         CALL PTUV( pswrite)('%Color: White on Black')
      ELSE
         CALL PTUV( pswrite)('%Color: Black on White [default]')
      END IF
 
      IF (LANDSCAPE) THEN
         CALL PTUV( pswrite)('%Page Style: Landscape')
      ELSE
         CALL PTUV( pswrite)('%Page Style: Portrait [default]')
      END IF
 
      CALL PTUV( pswrite)('%%EndProlog')
 
C
C   Determine the physical location of the image on the page.  The size of
C   The largest region to be plotted depends on the plotting mode.  The
C   defaults are:
C
C   Portrait:  6 x 6 inches maximum, centered on page with a 30pt binding
C              margin at left  [default orientation]
C
C   Landscape: 8 x 5.5 inches maximum, centered on page 
C
C   These may be changed using appropriate keywords (see preamble).
C
C   Note that sizes are expressed in rounded printers points (72 pt/inch).
C   In the following, I convert inches to points explicitly so as to keep
C   things reasonably transparent should future users wish to change things.
C
 
      IF (LANDSCAPE) THEN
         WPAGE = PLONG * 72.0
         HPAGE = PSHORT * 72.0
         WMAX = WWID * 72.0
         HMAX = WHEI * 72.0
      ELSE
         WPAGE = PSHORT * 72.0
         HPAGE = PLONG * 72.0
         WMAX = WWID * 72.0
         HMAX = WHEI * 72.0
      END IF
 
      IF ( NROW .GT. NCOL ) THEN
         HEIGHT = HMAX
         WIDTH = HMAX*FLOAT(NCOL)/FLOAT(NROW)
         IF (WIDTH .GT. WMAX) THEN
            WIDTH = WMAX
            HEIGHT = WMAX*FLOAT(NROW)/FLOAT(NCOL)
         END IF
      ELSE IF ( NROW .EQ. NCOL ) THEN
         WIDTH = MIN(WMAX,HMAX)
         HEIGHT = WIDTH
      ELSE IF ( NROW .LT. NCOL ) THEN
         WIDTH = WMAX
         HEIGHT = WMAX*FLOAT(NROW)/FLOAT(NCOL)
         IF (HEIGHT .GT. HMAX) THEN
            HEIGHT = HMAX
            WIDTH = HMAX*FLOAT(NCOL)/FLOAT(NROW)
         END IF
      END IF
 
C   Determine the location of the image origin (lower left-hand corner).
 
      IF (.NOT. UORIGIN) THEN
         HMARGIN = (HPAGE - HMAX) / 2.0
         WMARGIN = (WPAGE - WMAX) / 2.0
         W0 = (WMAX - WIDTH)/2.0 + WMARGIN
         H0 = (HMAX - HEIGHT)/2.0 + HMARGIN
      ELSE
         IF ((W0+WIDTH).GT. WPAGE) THEN
            PRINT *, '** Image overfills page horizontally'
            PRINT *, '** ... check ORIGIN and WIND'
            XERR = .TRUE.
            RETURN
         ELSE IF ((H0+HEIGHT) .GT. HPAGE) THEN
            PRINT *, '** Image overfills page vertically'
            PRINT *, '** ... check ORIGIN and WIND'
            XERR = .TRUE.
            RETURN
         END IF
      END IF
 
C   Add binding margin offset (includes 3.5em x 12pt labels) to portrait 
C   mode plots, but only if the origin has not been user defined.
 
      IF (.NOT. LANDSCAPE .AND. .NOT. UORIGIN .AND. .NOT. LARGE) 
     &          W0 = W0 + 30.0
 
C   Set the device coordinates of the graphics area
 
      WM = W0 + WIDTH
      HM = H0 + HEIGHT
 
C   If the image is scaled, set the axis limits appropriately.  Remember
C   that rows run top-to-bottom in VISTA images.  Add/Subtract a 1/2 pixel
C   bias because pixel coordinates refer to the *center* of the pixel, not
C   the space between pixels
 
      IF (IMSCALE) THEN
         RMIN = PSCALE*(RCEN - FLOAT(ER) + 0.5)
         RMAX = PSCALE*(RCEN - FLOAT(SR) - 0.5)
         CMIN = PSCALE*(FLOAT(SC) - CCEN - 0.5)
         CMAX = PSCALE*(FLOAT(EC) - CCEN + 0.5)
      ELSE
         RMIN = FLOAT(ER) + 0.5
         RMAX = FLOAT(SR) - 0.5
         CMIN = FLOAT(SC) - 0.5
         CMAX = FLOAT(EC) + 0.5        
      END IF

C  These are the X and Y sizes of the image to be rendered including binning.
C  This contortion ensures correct roundoff for the rastering.

      IXSIZE = NINT(FLOAT(NCOL)/FLOAT(IBIN))
      IYSIZE = INT(FLOAT(NROW)/FLOAT(IBIN))

C   Write out PostScript commands that setup reproducing the image
 
      CALL PTUV( pswrite) ('gsave')
      WRITE (PSCMD,'(1X,2(F9.3,2X),'' translate'')') W0, H0
      CALL PTUV( pswrite) (PSCMD)
      PSCMD=' '
      WRITE (PSCMD,'(1X,2(F9.3,2X),'' scale'')') WIDTH, HEIGHT
      CALL PTUV( pswrite) (PSCMD)
      PSCMD=' '
      WRITE (PSCMD,'(1X,''/ImStr '',i4,'' string def'')') IXSIZE
      CALL PTUV( pswrite) (PSCMD)
      PSCMD=' '
 
C   Dimensions of image in pixels, indicating 8 bit gray scale
 
      IF (FOUR) THEN
         WRITE (PSCMD,'(1X,2(I4,2X),'' 4'')') IXSIZE, IYSIZE
         NPERLINE = 80
         FMTSTR = '(  Z1)'
      ELSE
         WRITE (PSCMD,'(1X,2(I4,2X),'' 8'')') IXSIZE, IYSIZE
         NPERLINE = 40
         FMTSTR = '(  Z2.2)'
      END IF
      CALL PTUV( pswrite) (PSCMD)
      PSCMD=' '
 
C
C  Set the image transformation matrix depending on whether the image
C  pixels are scanned Left-to-Right/Top-to-Bottom ("standard" Video or
C  CCD rastering), or Left-to-Right/Bottom-to-Top ("Inverse" rastering,
C  or how IRAF and IDL do it).  Specified by the FLIP logical flag.
C
C  A note about PostScript transformation matrices, since this is rather
C  arcane...
C  PostScript uses 6 element transformation matrices with the following 
C  assignments:
C
C     [Xscale Xcross Ycross Yscale Xtranslate Ytranslate]
C
C  Such that:
C
C     X' = Xtranslate + Xscale*X + Xcross*Y    '
C     Y' = Ytranslate + Yscale*Y + Ycross*X    '
C
C  For an image scanned Left-Right/Top-Bottom, this becomes
C
C     [Xscale 0 0 -Yscale 0 Yscale]
C
C  We "invert" the Yscale so it runs 1->0 rather than 0->1, and translate
C  the origin to the upper left-hand corner (accomplished by doing no X-axis
C  translation, and a non-zero Y-axis translation).  Since there is no
C  rotation or shear, the cross terms (elements 2 & 3) are 0.  Note that the
C  image "box" used by PostScript has its own internal coordinate system in 
C  which the image is mapped into (X,Y) = (0:1,0:1) regardless of the image 
C  size OR axial ratio.  We enforce the 1:1 aspect ratio by changing the 
C  global X/Y axis scaling (see SCALE PostScript command issued above), which
C  stretches the 1x1 point box into the size and shape required.  Weird, but
C  that`s how the halftone imaging model works in PostScript.
C
C  For an image scanned Left-Right/Bottom-Top, the matrix is:
C
C     [Xscale 0 0 Yscale 0 0]
C
C  Here, there is no translation or rotation (or shear) required, and the 
C  image is painted in the usual PostScript sense (but inverse from "normal"
C  video raster order).
C
C     The power of this method becomes evident if we consider that we may
C  eventually need to rotate the image to align it with some relevant 'world'
C  coordinate system (like RA and DEC).  This is easily done internally by
C  the PostScript printer by setting the appropriate transformation matrix
C  elements - the first 4 in this case; simply the traditional sine/cosine
C  2-D rotation matrix elements, multiplied by the appropriate X and Y scale
C  factors - a considerable gain in programming simplicity as it obviates the
C  need for POSTIT to explicity rotate and re-bin the image array... the
C  printer does all the dirty work.  When POSTIT finally evolves to include
C  a contour overlay capability, things will be much simpler, as the LickMongo
C  contour routines have a similar transformation matrix built in.  All that
C  will be required is a clever 'visibility' algorithm to insure that
C  foreground contours don`t get lost against a dark (or light) background.
C
C  PostScript is your friend...
C

      IF (FLIP) THEN
         WRITE (PSCMD,'(1X,''['',I4,'' 0 0 '',I5,'' 0 0]'')')
     &        IXSIZE, IYSIZE
      ELSE
         WRITE (PSCMD,'(1X,''['',I4,'' 0 0 '',I5,'' 0 '',I4,'']'')')
     &        IXSIZE, -IYSIZE, IYSIZE
      END IF
      CALL PTUV( pswrite) (PSCMD)
      PSCMD=' '
 
      CALL PTUV( pswrite)('{currentfile ImStr readhexstring pop}')
      CALL PTUV( pswrite)('bind image')
 
C   Transform the image into [0,NLEV] using the desired mapping (Linear
C   or Histogram), convert each value into a HEXADECIMAL string and write 
C   into the current PostScript text file.
 
      F = SPAN / FLOAT(NLEV - 1)
      COMPF = FLOAT(IBIN*IBIN)
      ISTART = SR
      IEND = ER - IBIN + 1
      ISKIP = IBIN
      DO 8705 ROW = ISTART, IEND, ISKIP
         DO 8706 COL = SC, EC-IBIN+1, NPERLINE*IBIN
            DO 8707 K1 = 0, NPERLINE-1
               K = K1*IBIN
               IF ((COL + K) .GT. EC ) GO TO 100
               NDONE = K1 + 1
               APIX  = 0.0
 
C     Sum adjacent pixels if IBIN > 1
 
               DO 8708 IR = ROW, ROW+IBIN-1
                  DO 8709 IC = COL+K, COL+K+IBIN-1
                     APIX = APIX + A(IC,IR)
 8709             CONTINUE
 8708          CONTINUE
               APIX = (APIX/COMPF)
               IF (HISTEQ) THEN
                  IHST = NINT((APIX+BZERO)/BSCALE)
                  IHST = MAX(MIN(IHST,HEMAX),HEMIN)
                  IPIX(K1+1) = IMLUT(IHST)
               ELSE
                  APIX = MAX((APIX - ZERO),0.0)
                  IF (.NOT. CLIP) THEN
                     APIX = APIX - INT(APIX/SPAN) * SPAN
                     IPIX(K1+1) = INT( APIX / F )
                  ELSE
                     IPIX(K1+1) = INT ( APIX / F )
                     IF (IPIX(K1+1) .LT. 0) THEN
                        IPIX(K1+1) = 0
                     ELSE IF (IPIX(K1+1) .GT. NLEV-1) THEN
                        IPIX(K1+1) = NLEV-1
                     END IF
                  END IF
               END IF
               IF (.NOT. POSITIVE) IPIX(K1+1)=NLEV-1-IPIX(K1+1)
 8707       CONTINUE
 100        IF (NDONE .NE. 0) THEN
#ifdef __SUN3
               DO 9701 I = 1, NDONE
                  CALL VSHEX(IPIX(I),HH)
                  IF (FOUR) THEN
                     WRITE(PSCMD(I:I),'(A1)') HH(2:2)
                  ELSE
                     WRITE(PSCMD((I-1)*2+1:I*2),'(A2)') HH(1:2)
                  END IF
 9701          CONTINUE
#else
               WRITE ( FMTSTR(2:3), '(I2)' ) NDONE
               WRITE (PSCMD, FMTSTR ) (IPIX(J), J=1,NDONE)
#endif
               CALL PTUV( pswrite) (PSCMD)
               PSCMD=' '
            END IF
 8706    CONTINUE
 8705 CONTINUE
 
C   Finish the image
 
      CALL PTUV( pswrite) ('grestore')
 
C   Draw a 0 to NLEV gray scale wedge 
 
      IF (BAR) THEN
         XBAR = W0 + (WIDTH/2.) - 144.0
         XBARM = XBAR + 288
         IF (LANDSCAPE) THEN
            YBAR = H0 - 72.0
         ELSE
            YBAR = H0 - 90.0
         END IF
         YBARM = YBAR + 30
         CALL PTUV( pswrite)('gsave')
         WRITE (PSCMD,'(1X,F9.3,2X,F9.3,'' translate'')') XBAR, YBAR
         CALL PTUV( pswrite) (PSCMD)
         PSCMD=' '
         CALL PTUV( pswrite) (' 288 30 scale')
         IF (FOUR) THEN
            CALL PTUV( pswrite)('/barstr 16 string def')
            CALL PTUV( pswrite)('16 2 4 [16 0 0 2 0 0]')
            CALL PTUV( pswrite)('{currentfile barstr readhexstring pop}')
            CALL PTUV( pswrite)('bind image')
            IF (POSITIVE) THEN
               CALL PTUV( pswrite)('0123456789ABCDEF')
               CALL PTUV( pswrite)('0123456789ABCDEF')
            ELSE
               CALL PTUV( pswrite)('FEDCBA9876543210')
               CALL PTUV( pswrite)('FEDCBA9876543210')
            END IF
         ELSE
            CALL PTUV( pswrite)('/barstr 33 string def')
            CALL PTUV( pswrite)('33 2 8 [33 0 0 2 0 0]')
            CALL PTUV( pswrite)('{currentfile barstr readhexstring pop}')
            CALL PTUV( pswrite)('bind image')
            IF (NLEV .NE. 256) THEN
               ISTEP = NLEV/33
               IF (POSITIVE) THEN
                  DO 1011 I = 1, 33
                     IBAR(I) = ISTEP*(I-1)
 1011             CONTINUE
                  IBAR(33) = NLEV-1
               ELSE
                  DO 1012 I = 1,33 
                     IBAR(I) = (NLEV-1) - ISTEP*(I-1)
 1012             CONTINUE
                  IBAR(33) = 0
               END IF
#ifdef __SUN3
               DO 1012 I = 1, 33
                  CALL VSHEX(IBAR(I),HH)
                  WRITE(PSCMD((I-1)*2+1:I*2),'(A2)') HH(1:2)
 1012          CONTINUE
#else
               WRITE (PSCMD,'(33Z2.2)') (IBAR(I), I=1,33)
#endif
               CALL PTUV(pswrite)(PSCMD)
               CALL PTUV(pswrite)(PSCMD)
               PSCMD=' '
            ELSE
               IF (POSITIVE) THEN
               CALL PTUV( pswrite)(FSTRCAT('0008101820283038404850586068707880'
     &                                    ,'889098A0A8B0B8C0C8D0D8E0E8F0F8FF'))
               CALL PTUV( pswrite)(FSTRCAT('0008101820283038404850586068707880'
     &                                    ,'889098A0A8B0B8C0C8D0D8E0E8F0F8FF'))
               ELSE
               CALL PTUV( pswrite)(FSTRCAT('FFF8F0E8E0D8D0C8C0B8B0A8A098908880'
     &                                    ,'78706860585048403830282018100800'))
               CALL PTUV( pswrite)(FSTRCAT('FFF8F0E8E0D8D0C8C0B8B0A8A098908880'
     &                                    ,'78706860585048403830282018100800'))
               END IF
            END IF
         END IF
 
C   Finish the gray scale wedge
 
         CALL PTUV( pswrite) ('grestore')
 
C   Use LickMongo to draw a box around the gray scale wedge and label
C   it.  Label scale bar with extreme level limits (ZERO to ZERO+SPAN), 
C   unless a histogram equalization is being done, then the color bar is
C   unlabelled.

         CALL PMGO(SETLOC)(XBAR,YBAR,XBARM,YBARM)
         CALL PMGO(SETLWEIGHT)(1.0)
         IF (HISTEQ) THEN
            CALL PMGO(SETLIM)(0.,1.,0.,1.)
            CALL PMGO(ABOX)(5,5,5,5)
         ELSE
            BLMIN = ZERO
            BLMAX = ZERO+SPAN
            CALL PMGO(SETEXPAND)(0.75)
            CALL PMGO(SETLIM)(BLMIN,0.,BLMAX,1.)
            CALL PMGO(ABOX)(5,5,5,5)
            CALL PMGO(ABOX)(3,5,5,5)
            CALL PMGO(XLABEL)(LBAR,BARLAB)
	    CALL PMGO(SETEXPAND)(1.0)
         END IF
      END IF
 
C   Use LickMongo to draw a box around the image, label the axes, etc.
C   ** USEBOX added 95Jul3 [rwp] following Lowell version of JAH **

      CALL PMGO(SETLOC) (W0,H0,WM,HM)
      IF (USEBOX) THEN
         CALL PMGO(SETLWEIGHT)(1.0)
         IF (FLIP) THEN
            CALL PMGO(SETLIM)(CMIN,RMAX,CMAX,RMIN)
         ELSE
            CALL PMGO(SETLIM)(CMIN,RMIN,CMAX,RMAX)
         END IF
         IF (AXES) THEN
            CALL PMGO(BOX) (1,2)
            IF (IMSCALE) THEN
               CALL PMGO(XLABEL)(6,'Arcsec')
               CALL PMGO(YLABEL)(6,'Arcsec')
            ELSE
               CALL PMGO(XLABEL)(7,'Columns')
               CALL PMGO(YLABEL)(4,'Rows')
            END IF
         ELSE
            CALL PMGO(ABOX)(5,5,5,5)
         END IF
      END IF

C   If requested, put a title on the image
 
      IF (TITLE) THEN
	 IF (FLIP) THEN
	   YTITLE = RMIN - 0.1*(RMAX-RMIN)
	 ELSE
           YTITLE = RMAX + 0.1*(RMAX-RMIN)
	 END IF
         CALL PMGO(RELOCATE)(CMIN,YTITLE)
         CALL PMGO(SETEXPAND)(1.2)
         CALL PMGO(PUTLABEL)(LTITLE,IMTITLE,9)
         CALL PMGO(SETEXPAND)(1.0)
      END IF
 
C   If requested, put a comment on the image
 
      IF (COMMENT) THEN
	 IF (FLIP) THEN
           YTITLE = RMIN - 0.05*(RMAX-RMIN)
	 ELSE
           YTITLE = RMAX + 0.05*(RMAX-RMIN)
	 END IF
         CALL PMGO(RELOCATE)(CMIN,YTITLE)
         CALL PMGO(PUTLABEL) (LCOM,IMCOM,6)
      END IF

C   Enter interactive MONGO or execute a MONGO macro file, if requested

      IF (INTERACTIVE .OR. MACRO) THEN
        MONGOCOM(1) = 'DELETE 1 1000'
	MONGOCOM(2) = ' '
        IF (MACRO) THEN
	  MONGOCOM(3) = FSTRCAT('INPUT ',MACROFILE)
	ELSE
	  MONGOCOM(3) = ' '
	END IF
	IF (INTERACTIVE) THEN
	  MONGOCOM(4) = ' '
	ELSE
	  MONGOCOM(4) = 'END'
	END IF
        ZPLOT = 0.
	CALL PMGO(MONGO)(4,MONGOCOM,1,1,ZPLOT)
      END IF

C   If requested, write the info line starting in the lower left-hand corner

      IF (INFO) THEN
         WM = WPAGE - 18.0
         WH = HPAGE - 18.0
         CALL PMGO(SETLOC)(30.0,18.0,WM,WH)
         CALL PMGO(SETLIM)(0.,0.,1.,1.)
         CALL PMGO(RELOCATE)(0.,0.)
	 CALL PMGO(SETEXPAND) (0.66667)
         CALL PMGO(PUTLABEL) (LINFO,INFOLINE,9)
	 CALL PMGO(SETEXPAND) (1.0)
      END IF

C   Put in copypage commands if requested

      DO 7706 I = 2, NCOPIES
         CALL PTUV( pswrite)('copypage')
 7706 CONTINUE

C   Terminate the PostScript file.
 
      IVEC = PMGO(FILEPLOT)(0)
 
C   Good Bye!
 
      RETURN
      END
 
C-----------------------------------------------------------------------------
 
      SUBROUTINE NUMBSTR (MM, PP, FORM, STRING, NC)
 
C
C  NUMBSTR:  Convert a number into a LickMongo character string
C
C Adapted from PGNUMB by Tim Pearson
C
C This routine converts a number into a decimal character representation. To
C avoid problems of floating-point roundoff, the number must be provided as an
C integer (MM) multiplied by a power of 10 (10**PP).  The output string
C retains only significant digits of MM, and will be in either integer format
C (123), decimal format (0.0123), or exponential format (1.23x10**5).  The 
C standard LickMONGO escape sequence \\u is used to raise the exponent and 
C \g* is used for the multiplication sign.
C
C ----------
C
C Formatting rules:
C   (a) Decimal notation (FORM=1):
C       - Trailing zeros to the right of the decimal sign are
C         omitted
C       - The decimal sign is omitted if there are no digits
C         to the right of it
C       - When the decimal sign is placed before the first digit
C         of the number, a zero is placed before the decimal sign
C       - The decimal sign is a period (.)
C       - No spaces are placed between digits (ie digits are not
C         grouped in threes as they should be)
C       - A leading minus (-) is added if the number is negative
C   (b) Exponential notation (FORM=2):
C       - The exponent is adjusted to put just one (non-zero)
C         digit before the decimal sign
C       - The mantissa is formatted as in (a), unless its value is
C         1 in which case it and the multiplication sign are omitted
C       - If the power of 10 is not zero and the mantissa is not
C         zero, an exponent of the form \g*10\\u[-]nnn is appended,
C         where \g* is a multiplication sign (cross), \\u is an escape
C         sequence to raise the exponent, and as many digits nnn
C         are used as needed
C   (c) Automatic choice (FORM=0):
C         Decimal notation is used if the absolute value of the
C         number is less than 10000 or greater than or equal to
C         0.01. Otherwise exponential notation is used.
C
C Arguments:
C
C  MM     (input)  : Mantissa of the value to be formatted
C  PP     (input)  : Exponent of the value to be formatted
C
C      full value to be formatted is: MM*10**PP.
C
C  FORM   (input)  : controls how the number is formatted:
C                    FORM = 0 -- use either decimal or exponential
C                    FORM = 1 -- use decimal notation
C                    FORM = 2 -- use exponential notation
C  STRING (output) : the formatted character string, left justified.
C                    If the length of STRING is insufficient, a single
C                    asterisk is returned, and NC=1.
C  NC     (output) : the number of characters used in STRING:
C                    the string to be printed is STRING(1:NC).
C
C-----------------------------------------------------------------------
 
C   External Variables
 
      INTEGER MM, PP, FORM
      CHARACTER*(*) STRING
      INTEGER NC
 
C   Internal Variables
 
      CHARACTER*20 WORK, WEXP, TEMPWORK
      CHARACTER FSTRCAT*80
      INTEGER M, P, ND, I, J, K, NBP
      LOGICAL MINUS
 
C   Zero is always printed as `0`.
 
      IF (MM.EQ.0) THEN
         STRING = '0'
         NC = 1
         RETURN
      END IF
 
C   If negative, make a note of that fact.
 
      MINUS = MM.LT.0
      M = ABS(MM)
      P = PP
 
C   Convert M to a left-justified digit string in WORK. As M is a
C   positive integer, it cannot use more than 10 digits (2147483647).
 
      J = 10
8712  IF (M.NE.0) THEN
         K = MOD(M,10)
         M = M/10
         WORK(J:J) = CHAR(ICHAR('0')+K)
         J = J-1
         GO TO 8712
      END IF
      WORK = WORK(J+1:)
      ND = 10-J
 
C   Remove right-hand zeros, and increment P for each one removed.
C   ND is the final number of significant digits in WORK, and P the
C   power of 10 to be applied. Number of digits before decimal point
C   is NBP.
 
8713  IF (WORK(ND:ND).EQ.'0') THEN
         ND = ND-1
         P = P+1
         GO TO 8713
      END IF
      NBP = ND+MIN(P,0)
 
C   Integral numbers of 4 or less digits are formatted as such.
 
      IF ((P.GE.0) .AND. (P+ND.LE.4)) THEN
         DO 8714 I=1,P
            ND = ND+1
            WORK(ND:ND) = '0'
8714     CONTINUE
         P = 0
 
C   If NBP is 4 or less, simply insert a decimal point in the right place.
 
      ELSE IF (NBP.GE.1.AND.NBP.LE.4.AND.NBP.LT.ND) THEN
         WORK(NBP+2:ND+1) = WORK(NBP+1:ND)
         WORK(NBP+1:NBP+1) = '.'
         ND = ND+1
         P = 0
 
C   Otherwise insert a decimal point after the first digit, and adjust P.
 
      ELSE
         P = P + ND - 1
         IF (P.EQ.-1) THEN
            TEMPWORK = FSTRCAT('0',WORK)
            WORK = TEMPWORK
            ND = ND+1
            P = 0
         ELSE IF (P.EQ.-2) THEN
            TEMPWORK = FSTRCAT('00',WORK)
            WORK = TEMPWORK
            ND = ND+2
            P = 0
         END IF
         IF (ND.GT.1) THEN
            WORK(3:ND+1) = WORK(2:ND)
            WORK(2:2) = '.'
            ND = ND + 1
         END IF
      END IF
 
C   Add exponent if necessary (LickMongo LABEL or PUTLABEL notation)
 
      IF (P.NE.0) THEN
#ifdef DOUBLE_BACKSLASH
         WORK(ND+1:ND+8) = '\\g*10\\\\u'
#else
         WORK(ND+1:ND+8) = '\g*10\\u'
#endif
         ND = ND+8
         IF (P.LT.0) THEN
            P = -P
            ND = ND+1
            WORK(ND:ND) = '-'
C         ELSE
C            ND = ND + 1
C            WORK(ND:ND) = '+'
         END IF
         J = 10
8715     IF (P.NE.0) THEN
            K = MOD(P,10)
            P = P/10
            WEXP(J:J) = CHAR(ICHAR('0')+K)
            J = J-1
            GO TO 8715
         END IF
         WORK(ND+1:) = WEXP(J+1:10)
         ND = ND+10-J
         IF (WORK(1:2).EQ.'1E') THEN
            WORK = WORK(3:)
            ND = ND-2
         END IF
#ifdef DOUBLE_BACKSLASH
         WORK(ND+1:ND+3) = '\\d'
#else
	 WORK(ND+1:ND+3) = '\\\\d'
#endif
 	 ND = ND+3
      END IF
 
C   Add minus sign if necessary and move result to output.
 
      IF (MINUS) THEN
         STRING = FSTRCAT('-',WORK(1:ND))
         NC = ND+1
      ELSE
         STRING = WORK(1:ND)
         NC = ND
      END IF
 
C   Check result fits.
 
      IF (NC.GT.LEN(STRING)) THEN
         STRING = '*'
         NC = 1
      END IF
 
      RETURN
      END

C******************************************************************************
#ifdef __SUN3
      SUBROUTINE VSHEX(I,HH)

C   Convert an integer into hex
	
      CHARACTER*2 HH

      J = I - (I/16)*16
      IND = ICHAR('a')
      ISTART = 1
      IF (I .GT. 15) THEN
	K = (I - J) / 16
	IF (K .LT. 10) THEN
	  WRITE(HH(1:1),'(I1)') K
	ELSE
	  HH(1:1) = CHAR(IND+(K-10))
	END IF
      ELSE
	HH(1:1) = '0'
      END IF
      IF (J .LT. 10) THEN
	WRITE(HH(2:2),'(I1)') J
      ELSE
	HH(2:2) = CHAR(IND+(J-10))
      END IF

      RETURN
      END
#endif

