#include "Vista.h"
C       Compute the heliocentric julian date and related parameters

        SUBROUTINE GETHJD(HEADER)

C       Input:  HEADER = A Lick style FITS header buffer

C       Output: Parameters in common block TIMEDATA.  The parameters,
C               which are all DOUBLE PRECISION, are computed for the
C               mid-point of the observation.  The midpoint is computed
C               by adding one half the exposure time to the starting
C               time of the observation.  Hence, no account is taken
C               of elapsed time in a 'PAUSED' observation.
C               The parameters are:
C               JD      = Julian date of the observation.
C               HJD     = Heliocentric julian date of the observation
C               LST     = Local siderial time
C               HA      = Hour angle (computed, not from header)
C               Z       = Zenith distance
C               AM      = Airmass
C               VCORR   = Heliocentric velocity correction
C               MONTH   = UT month of observation
C               DAY     = UT day of observation
C               YEAR    = UT year of observation
C               UT      = UT start time in seconds
C               LATITUDE= Latitude of the observatory

C       Written by:     Richard Stover
C                       Lick Observatory
C                       University of California
C                       Santa Cruz, CA 95064

C                       December, 1983

        CHARACTER HEADER*(*)

#ifdef VMS
        INCLUDE 'VINCLUDE:TIMEDATA.INC'        ! Output parameter definitions
#else
        INCLUDE 'vistadisk/source/include/timedata.inc'
#endif
        CHARACTER*14 CUTDATE,CUTTIME
        INTEGER UPPER
        DOUBLE PRECISION FHEAD

        PARAMETER (PI=3.1415926536D0)

C  Get exposure, RA and DEC. This code assumes RA and DEC are stored
C    as characters, hh:mm:ss and dd:mm:ss
        EXPOSURE = FLOAT(INHEAD('EXPOSURE',HEADER))
        CALL CHEAD('RA',HEADER,CUTTIME)
        IF (CUTTIME .EQ. ' ') GOTO 9400
        CALL GETCOORD(CUTTIME,IHR,IMIN,SEC,SG)
        RA = SG * (IHR + IMIN/60. + SEC/3600.) * 15. * PI / 180.

        CALL CHEAD('DEC',HEADER,CUTTIME)
        IF (CUTTIME .EQ. ' ') GOTO 9400
        CALL GETCOORD(CUTTIME,IHR,IMIN,SEC,SG)
        DEC = SG * (IHR + IMIN/60. + SEC/3600.) * PI / 180.

C  Hour angle will be computed by HELJD, but we read the card here
C       in the case that only the HA, rather than the UT is available.
C       In this case, HELJD will use the given HA
        CALL CHEAD('HA',HEADER,CUTTIME)
        HA = 0.
        IF (CUTTIME .NE. ' ') THEN
          CALL GETCOORD(CUTTIME,IHR,IMIN,SEC,SG)
          HA = SG * (IHR + IMIN/60. + SEC/3600.) * 15. * PI / 180.
        END IF

C   These next 2 lines are commented out. They would apply if RA and DEC were
C     stored as real numbers in units of seconds of time or arc.
C        DEC = FHEAD('DEC',HEADER)*2.0*PI/(360.0*60.0*60.0)
C        RA = FHEAD('RA',HEADER)*2.0*PI/(24.0*60.0*60.0)

C   Get UT date, assuming its in DATE-OBS as a character string
        CALL CHEAD('DATE-OBS',HEADER,CUTDATE)
        IF (CUTDATE .EQ. ' ') GOTO  9400
        L = UPPER(CUTDATE)
        DO 8701 I=1,L
                IF(CUTDATE(I:I) .EQ. '/') CUTDATE(I:I) = ' '
8701    CONTINUE
        CALL GETCOORD(CUTDATE,ND,MONTH,YEAR,SG)
        IF (YEAR .LT. 100.) YEAR = YEAR + 1900.0
        DAY = FLOAT(ND)

C   Get UT Start time from TIME card, assuming its a character string.
C    Add 0.5*EXPOSURE to the start time to use the time at mid-observation.
        CALL CHEAD('TIME',HEADER,CUTTIME)
        IF (CUTTIME .EQ. ' ' .AND. HA .EQ. 0.) GOTO  9400
        UT = 0.
        IF (CUTTIME .NE. ' ') THEN
          CALL GETCOORD(CUTTIME,NH,NM,FS,SG)
          UT = DFLOAT(NH) + DFLOAT(NM)/60.0 + (FS + 0.5*EXPOSURE)/3600.0
        END IF

C   Get the airmass from the header if it exists. We wont use this, but will
C     use it as a check to warn user if wrong longitude latitude is set!
        AM = FHEAD('AM',HEADER)
        IF (AM .EQ. 0) AM = FHEAD('AIRMASS',HEADER)
        IF (AM .EQ. 0) THEN
          AM = FHEAD('ZD',HEADER)
          IF (AM .GT. 0) AM = 1 / COSD(AM)
        END IF

C   Do the calculations.
        CALL HELJD(RA,DEC,MONTH,DBLE(DAY),DBLE(YEAR),
     &          UT,JD,HJD,LST,HA,Z,AM,VCORR,LATITUDE)

        RETURN
9400    PRINT *, ' One of the following cards is missing from the ',
     &           'FITS  header: '
        PRINT *, '   RA,  DEC,   UT,   DATE-OBS '
        PRINT *, ' Can''t compute airmass / Julian date '
        JD = 0.
        HJD = 0.
        LST = 0.
        HA = 0.
        Z = 0.
        AM = 0.
        RETURN

        END
